package com.moloco.sdk.internal.services.events

import android.net.Uri
import com.moloco.sdk.UserIntent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.services.AdData
import com.moloco.sdk.internal.services.AdDataService
import com.moloco.sdk.internal.services.AppInfo
import com.moloco.sdk.internal.services.AppInfoService
import com.moloco.sdk.internal.services.DeviceInfo
import com.moloco.sdk.internal.services.DeviceInfoService
import com.moloco.sdk.internal.services.NetworkInfo
import com.moloco.sdk.internal.services.NetworkInfoService
import com.moloco.sdk.internal.services.ScreenInfo
import com.moloco.sdk.internal.services.ScreenInfoService
import com.moloco.sdk.internal.services.proto.ProtoEncoderService
import com.moloco.sdk.internal.services.usertracker.UserTrackerService
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventConfigService

private const val TAG = "CustomUserEventBuilderServiceImpl"
internal const val QUERY_PARAMETER_NAME = "user_ad_interaction_ext"

/**
 * A custom user event builder service implementation that is provided by the Moloco SDK
 */
internal class CustomUserEventBuilderServiceImpl(
    val appInfoService: AppInfoService,
    val networkInfoService: NetworkInfoService,
    val deviceInfoService: DeviceInfoService,
    val screenInfoService: ScreenInfoService,
    val userIdentifierService: UserTrackerService,
    val adDataService: AdDataService,
    val encoderService: ProtoEncoderService,
    val userEventConfigService: CustomUserEventConfigService,
    val sdkVersion: String,
) : CustomUserEventBuilderService {

    override suspend fun userAdInteractionExtAsQueryParameter(
        eventTimestamp: Long,
        interaction: CustomUserEventBuilderService.UserInteraction,
        url: String
    ): String {
        if (!userEventConfigService.isCustomUserEventReportingEnabled()) {
            MolocoLogger.debug(
                TAG,
                "Event reporting config disabled, UserAdInteractionExt not reporting"
            )
            return url
        }

        val userAdInteractionExt = userAdInteractionExt(eventTimestamp, interaction)
        return Uri.parse(
            url
        ).buildUpon().appendQueryParameter(
            QUERY_PARAMETER_NAME,
            userAdInteractionExt
        ).build().toString()
    }

    private suspend fun userAdInteractionExt(
        eventTimestamp: Long,
        interaction: CustomUserEventBuilderService.UserInteraction
    ): String {
        val userAdInteractionExtBuilder = UserIntent.UserAdInteractionExt.newBuilder().apply {
            if (userEventConfigService.isUserTrackingEnabled()) {
                mref = userIdentifierService.getIdentifier()
            }
            when (val adData = adDataService.advertisingData()) {
                is AdData.Available -> {
                    advertisingId = adData.id
                }
                is AdData.Unavailable -> { /* NO-OP */ }
            }
            clientTimestamp = eventTimestamp
            addSdkMetadata()
            addInteraction(interaction, screenInfoService())
            addAppMetadata(appInfoService())
            addDeviceMetadata(deviceInfoService())
            addNetworkMetadata(networkInfoService())
        }

        val userAdInteractionExt = userAdInteractionExtBuilder.build()
        MolocoLogger.debug(TAG, "Encoding protobuf UserAdInteractionExt: $userAdInteractionExt")
        val userAdInteractionExtBuilderBase64 = encoderService.encodeUserAdInteractionExt(
            userAdInteractionExt
        )
        MolocoLogger.debug(
            TAG,
            "Successfully built userAdInteractionExt as base64 string: $userAdInteractionExtBuilderBase64"
        )

        return userAdInteractionExtBuilderBase64
    }

    private fun UserIntent.UserAdInteractionExt.Builder.addSdkMetadata() = this.apply {
        sdk = UserIntent.UserAdInteractionExt.MolocoSDK.newBuilder().apply {
            coreVer = sdkVersion
        }.build()
    }

    private fun UserIntent.UserAdInteractionExt.Builder.addDeviceMetadata(
        deviceInfo: DeviceInfo
    ): UserIntent.UserAdInteractionExt.Builder = this.apply {
        device = UserIntent.UserAdInteractionExt.Device.newBuilder().apply {
            osVer = deviceInfo.osVersion
            model = deviceInfo.model
            os = UserIntent.UserAdInteractionExt.Device.OsType.ANDROID
            screenScale = deviceInfo.screenDensity
        }.build()
    }

    private fun UserIntent.UserAdInteractionExt.Builder.addNetworkMetadata(
        networkInfo: NetworkInfo
    ): UserIntent.UserAdInteractionExt.Builder = this.apply {
        network = UserIntent.UserAdInteractionExt.Network.newBuilder().apply {
            when (networkInfo) {
                is NetworkInfo.CellularNetwork -> {
                    connectionType =
                        UserIntent.UserAdInteractionExt.Network.ConnectionType.CELLULAR
                    carrier = networkInfo.carrier
                }
                NetworkInfo.NoNetwork -> {
                    connectionType =
                        UserIntent.UserAdInteractionExt.Network.ConnectionType.UNKNOWN
                }
                NetworkInfo.WifiNetwork -> {
                    connectionType =
                        UserIntent.UserAdInteractionExt.Network.ConnectionType.WIFI
                }
            }
        }.build()
    }

    private fun UserIntent.UserAdInteractionExt.Builder.addAppMetadata(
        appInfo: AppInfo
    ): UserIntent.UserAdInteractionExt.Builder = this.apply {
        app = UserIntent.UserAdInteractionExt.App.newBuilder().apply {
            id = appInfo.packageName
            ver = appInfo.version
        }.build()
    }

    private fun UserIntent.UserAdInteractionExt.Builder.addInteraction(
        interaction: CustomUserEventBuilderService.UserInteraction,
        screenInfo: ScreenInfo
    ): UserIntent.UserAdInteractionExt.Builder {
        when (interaction) {
            is CustomUserEventBuilderService.UserInteraction.Impression -> {
                impInteraction = UserIntent.UserAdInteractionExt.ImpressionInteraction.newBuilder()
                    .build()
            }
            is CustomUserEventBuilderService.UserInteraction.Click -> {
                clickInteraction = UserIntent.UserAdInteractionExt.ClickInteraction.newBuilder()
                    .apply {
                        clickPos = interaction.clickPosition.toProtoPosition()
                        screenSize = screenInfo.toProtoSize()

                        interaction.viewSize?.let {
                            viewSize = it.toProtoSize()
                        }

                        interaction.viewPosition?.let {
                            viewPos = it.toProtoPosition()
                        }

                        addAllButtons(
                            interaction.buttonLayout.map { layoutButton ->
                                UserIntent.UserAdInteractionExt.Button.newBuilder().apply {
                                    type = layoutButton.buttonType.toProtoEnum()
                                    pos = layoutButton.position.toProtoPosition()
                                    size = layoutButton.size.toProtoSize()
                                }.build()
                            }
                        )
                    }.build()
            }
            is CustomUserEventBuilderService.UserInteraction.AppForeground -> {
                appForegroundingInteraction = UserIntent.UserAdInteractionExt.AppForegroundingInteraction.newBuilder().apply {
                    bgTsMs = interaction.lastBgTimestamp
                }.build()
            }
            is CustomUserEventBuilderService.UserInteraction.AppBackground -> {
                appBackgroundingInteraction = UserIntent.UserAdInteractionExt.AppBackgroundingInteraction.newBuilder().build()
            }
        }
        return this
    }
}

internal fun CustomUserEventBuilderService.UserInteraction.Button.ButtonType.toProtoEnum(): UserIntent.UserAdInteractionExt.Button.Type {
    return when (this) {
        CustomUserEventBuilderService.UserInteraction.Button.ButtonType.NONE -> UserIntent.UserAdInteractionExt.Button.Type.NONE
        CustomUserEventBuilderService.UserInteraction.Button.ButtonType.CLOSE -> UserIntent.UserAdInteractionExt.Button.Type.CLOSE
        CustomUserEventBuilderService.UserInteraction.Button.ButtonType.SKIP -> UserIntent.UserAdInteractionExt.Button.Type.SKIP
        CustomUserEventBuilderService.UserInteraction.Button.ButtonType.SKIP_DEC -> UserIntent.UserAdInteractionExt.Button.Type.DEC_SKIP
        CustomUserEventBuilderService.UserInteraction.Button.ButtonType.MUTE -> UserIntent.UserAdInteractionExt.Button.Type.MUTE
        CustomUserEventBuilderService.UserInteraction.Button.ButtonType.UNMUTE -> UserIntent.UserAdInteractionExt.Button.Type.UNMUTE
        CustomUserEventBuilderService.UserInteraction.Button.ButtonType.CTA -> UserIntent.UserAdInteractionExt.Button.Type.CTA
        CustomUserEventBuilderService.UserInteraction.Button.ButtonType.REPLAY -> UserIntent.UserAdInteractionExt.Button.Type.REPLAY
        CustomUserEventBuilderService.UserInteraction.Button.ButtonType.AD_BADGE -> UserIntent.UserAdInteractionExt.Button.Type.AD_BADGE
    }
}

internal fun CustomUserEventBuilderService.UserInteraction.Position.toProtoPosition(): UserIntent.UserAdInteractionExt.Position {
    return UserIntent.UserAdInteractionExt.Position.newBuilder().apply {
        x = topLeftXDp
        y = topLeftYDp
    }.build()
}

internal fun CustomUserEventBuilderService.UserInteraction.Size.toProtoSize(): UserIntent.UserAdInteractionExt.Size {
    return UserIntent.UserAdInteractionExt.Size.newBuilder().apply {
        w = widthDp
        h = heightDp
    }.build()
}

internal fun ScreenInfo.toProtoSize(): UserIntent.UserAdInteractionExt.Size {
    return UserIntent.UserAdInteractionExt.Size.newBuilder().apply {
        w = screenWidthDp
        h = screenHeightDp
    }.build()
}
