package com.moloco.sdk.internal.services.init

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.Init.SDKInitResponse
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmResultTag
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.publisher.MediationInfo
import io.ktor.http.HttpStatusCode
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

internal typealias SDKInitResult = Result<SDKInitResponse, InitFailure>

/**
 * Stateful service that provides capability to invoke init API the SDK as well as cache
 * the last successful init response
 */
internal interface InitService {
    /**
     * @return null if the SDK hasn't been initialized. non-null if SDK has successfully init
     */
    val currentSessionInitResponse: SDKInitResponse?

    /**
     * @param appKey [String] required for Moloco SDK
     * @param mediationInfo [MediationInfo][com.moloco.sdk.publisher.MediationInfo] required for Moloco SDK, when running in not-direct-integration/bid-token mode.
     * @return Performs init with backend and returns the init response
     *
     * NOTE: Caller has to ensure that performInit is only called once at a time.
     * This is not thread safe. We have top level thread safety
     */
    suspend fun performInit(
        appKey: String,
        mediationInfo: MediationInfo,
    ): Result<SDKInitResponse, InitFailure>

    /**
     * Clears the state of the service
     */
    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    suspend fun clearState()
}


internal data class FetchState(val sdkInitResult: SDKInitResult, val fetchType: String)
private const val TAG = "InitService"
internal class InitServiceImpl(private val initApi: InitApi,
                               private val initCache: InitCache,
                               private val scope: CoroutineScope,
) : InitService {
    private var _currentSessionInitResponse: SDKInitResponse? = null
    override val currentSessionInitResponse: SDKInitResponse?
        get() = _currentSessionInitResponse

    override suspend fun performInit(
        appKey: String,
        mediationInfo: MediationInfo,
    ): SDKInitResult {
        val performInitEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.SDKPerformInitAttempt.eventName)
        val fetchState = fetchInitResponse(appKey, mediationInfo)
        when(fetchState.sdkInitResult) {
            is Result.Failure -> {
                AndroidClientMetrics.recordCountEvent(
                    CountEvent(AcmCount.SDKPerformInitAttempt.eventName).withTag(AcmTag.Result.tagName, AcmResultTag.failure.name).withTag("state", fetchState.fetchType)
                )

                AndroidClientMetrics.recordTimerEvent(performInitEvent.withTag(AcmTag.Result.tagName, AcmResultTag.failure.name).withTag("state", fetchState.fetchType))
            }
            is Result.Success -> {
                AndroidClientMetrics.recordCountEvent(
                    CountEvent(AcmCount.SDKPerformInitAttempt.eventName).withTag(AcmTag.Result.tagName, AcmResultTag.success.name).withTag("state", fetchState.fetchType)
                )
                AndroidClientMetrics.recordTimerEvent(performInitEvent.withTag(AcmTag.Result.tagName, AcmResultTag.success.name).withTag("state", fetchState.fetchType))
            }
        }
        return fetchState.sdkInitResult
    }

    private suspend fun fetchInitResponse(appKey: String, mediationInfo: MediationInfo): FetchState {
        _currentSessionInitResponse?.let {
            MolocoLogger.info(TAG, "Returning current session init response")
            return FetchState(Result.Success(it), "in_memory")
        }

        // _currentSessionInitResponse is null, so check from initCache
        val cachedInitResponse = initCache.get(CacheKey(appKey, mediationInfo.name))
        if (cachedInitResponse != null) {
            MolocoLogger.info(TAG, "Returning cached init response")
            _currentSessionInitResponse = cachedInitResponse
            scope.launch {
                MolocoLogger.info(TAG, "Async fetching init response")
                fetchServerInitResponse(appKey, mediationInfo, true)
            }
            return FetchState(Result.Success(cachedInitResponse), "cache")
        }

        // No cached init response, so fetch from server
        MolocoLogger.info(TAG, "No cached response, fetching from server")
        val fetchResult = fetchServerInitResponse(appKey, mediationInfo, false)
        when(fetchResult) {
            is Result.Success -> {
                _currentSessionInitResponse = fetchResult.value
            }
            is Result.Failure -> {
                /* NO OP - Don't update any state */
                MolocoLogger.info(TAG, "Fetching init response failed")
            }
        }
        return FetchState(fetchResult, "network")
    }

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    override suspend fun clearState() {
        _currentSessionInitResponse = null
        initCache.clearAll()
    }

    internal suspend fun fetchServerInitResponse(appKey: String,
                                                 mediationInfo: MediationInfo,
                                                 asyncFetch: Boolean): SDKInitResult {
        lateinit var result: Result<SDKInitResponse, InitFailure>

        // Try initializing up to 3 times
        // https://docs.google.com/document/d/1EqQfiptDtcgjwWNET4FuYjQSqUXsxcCbT1JJY67veqA/edit#heading=h.iidd1pqa86oj
        repeat(3) { attempt ->
            result = initApi(appKey, mediationInfo = mediationInfo)

            when (result) {
                is Result.Success -> // If initialization succeeds, return early from the coroutine
                    (result as Result.Success<SDKInitResponse, InitFailure>).value.let {
                        MolocoLogger.info(TAG, "Init, successful in attempt(#$attempt)")
                        val cacheKey = CacheKey(appKey, mediationInfo.name)
                        with(initCache) {
                            MolocoLogger.info(TAG, "Clearing cache for old init response")
                            clearCache(cacheKey)
                            MolocoLogger.info(TAG, "Updating cache to new init response")
                            updateCache(cacheKey, it)
                        }

                        AndroidClientMetrics.recordCountEvent(
                            CountEvent(AcmCount.SDKFetchInitAttempt.eventName)
                                .withTag(AcmTag.Result.tagName, AcmResultTag.success.name)
                                .withTag(AcmTag.RetryAttempt.tagName, "$attempt")
                                .withTag("async", asyncFetch.toString())
                        )
                        return@fetchServerInitResponse Result.Success(it)
                    }
                is Result.Failure -> (result as Result.Failure<SDKInitResponse, InitFailure>).let {
                    val failureReason = when(val errorType = it.value) {
                        is InitFailure.ClientError -> errorType.type
                        is InitFailure.ServerError -> errorType.statusCode.toString()
                    }
                    AndroidClientMetrics.recordCountEvent(
                        CountEvent(AcmCount.SDKFetchInitAttempt.eventName)
                            .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                            .withTag(AcmTag.RetryAttempt.tagName, "$attempt")
                            .withTag(AcmTag.Reason.tagName, "$failureReason")
                            .withTag("async", asyncFetch.toString())
                    )
                    MolocoLogger.info(
                        TAG,
                        "Init attempt(#$attempt) failed with error: $failureReason"
                    )

                    // If the error is non-retryable, short-circuit the retry loop
                    if (!it.isRetryable()) {
                        if (it.value is InitFailure.ServerError) {
                            MolocoLogger.error(
                                TAG,
                                "Init response is non-retryable server failure: ${it.value.statusCode}, clearing cache"
                            )
                            // For non-retryable server errors, clear the cache. We may want to clear the
                            // cache for next session if the app is no longer valid (for eg - 403/404 errors)
                            initCache.clearCache(CacheKey(appKey, mediationInfo.name))
                        } else {
                            MolocoLogger.error(
                                TAG,
                                "Init response is non-retryable server or client failure: ${it.value}"
                            )
                        }
                        return result
                    }
                }
            }

            // Otherwise, wait for 1 second before retrying
            delay(1000)
        }

        MolocoLogger.info(TAG, "Moloco SDK Init failed after all retries: $result")
        return result
    }
}

internal fun Result.Failure<SDKInitResponse, InitFailure>.isRetryable(): Boolean {
    // Only select server errors are retryable
    if (this.value is InitFailure.ServerError) {
        return this.value.statusCode.isRetryable()
    }

    // All client errors are retryable
    return true
}

internal fun Int.isRetryable(): Boolean = this == HttpStatusCode.TooManyRequests.value ||
        this == HttpStatusCode.RequestTimeout.value || this < 400 || this >= 500
