@file:Suppress("unused")

package com.moloco.sdk.internal.unity_bridge

import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.android_context.ApplicationContext
import com.moloco.sdk.internal.unity_bridge.internal.UnityAdManager
import com.moloco.sdk.publisher.Initialization
import com.moloco.sdk.publisher.MediationInfo
import com.moloco.sdk.publisher.Moloco
import com.moloco.sdk.publisher.init.MolocoInitParams
import kotlinx.coroutines.flow.MutableStateFlow

/**
 * Bridge class that enables Unity applications to initialize the Moloco SDK.
 *
 * This class is marked as internal to prevent direct usage from regular JVM code,
 * but remains accessible via reflection from Unity's C# environment. Unity uses
 * reflection to call the static methods in this class to interface with the native
 * Android SDK.
 */
internal object MolocoUnityPlugin {
    private val initState = MutableStateFlow(false);
    private val context by lazy { ApplicationContext() }
    private val adManager by lazy { UnityAdManager() }

    /**
     * Initializes the Moloco SDK with the provided parameters.
     *
     * This method handles the initialization process for Unity applications by
     * configuring the SDK with the application context, developer app key, and
     * the Unity mediation information.
     *
     * @param appKey The developer app key provided by Moloco
     * @param mediationName The name of the mediation platform being used
     * @param callback Callback interface to notify Unity of initialization result
     */
    @JvmStatic
    fun initializeSdk(appKey: String, mediationName: String, callback: MolocoUnityInitCallback) {
        Moloco.initialize(MolocoInitParams(context, appKey, MediationInfo(mediationName))) { initStatus ->
            MolocoLogger.info(TAG,"initialized $appKey $mediationName ${initStatus.initialization}")

            (initStatus.initialization == Initialization.SUCCESS).let {
                initState.value = it
                callback.onInitialized(it)
            }
        }
    }

    /**
     * Loads an interstitial ad for the specified ad unit.
     *
     * This method creates and loads an interstitial ad with the provided bid response.
     * It delegates to the UnityAdManager for the actual loading process.
     * The method will return early with an error if the SDK has not been initialized.
     *
     * @param adUnitId The unique identifier for the ad unit
     * @param bidResponse The bid response JSON string containing ad content details
     * @param callback Callback interface to notify Unity of ad load events
     */
    @JvmStatic
    fun loadInterstitial(adUnitId: String, bidResponse:String, callback: MolocoUnityLoadCallback) {
        if (!initState.value) return logUninitializedAccessError("loadInterstitial")
        adManager.loadInterstitial(adUnitId.trim(), bidResponse, callback)
    }

    /**
     * Shows a previously loaded interstitial ad.
     *
     * Attempts to display the ad identified by the provided ad unit ID.
     * It delegates to the UnityAdManager for the actual display process.
     * The method will return early with an error if the SDK has not been initialized.
     *
     * @param adUnitId The unique identifier for the ad unit to display
     * @param callback Callback interface to notify Unity of ad display events
     */
    @JvmStatic
    fun showInterstitial(adUnitId: String, callback: MolocoUnityShowCallback) {
        if (!initState.value) return logUninitializedAccessError("showInterstitial")
        adManager.showInterstitial(adUnitId.trim(), callback)
    }

    /**
     * Logs an error when a method is called before SDK initialization.
     *
     * This helper method provides consistent error logging when methods
     * are called before the SDK has been properly initialized.
     *
     * @param callingMethod The name of the method that was called prematurely
     */
    private fun logUninitializedAccessError(callingMethod: String) {
        MolocoLogger.error(TAG, "ERROR: Failed to execute $callingMethod() - please ensure the Moloco Unity Plugin has been initialized by calling 'MolocoSdk.InitializeSdk();'!")
    }

    private const val TAG = "MolocoUnityPlugin"
}
