package com.moloco.sdk.internal.utils

import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.CoroutineStart
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.coroutines.withContext
import kotlin.coroutines.CoroutineContext
import kotlin.coroutines.EmptyCoroutineContext
import kotlin.coroutines.coroutineContext

/**
 * A wrapper around [CoroutineScope.launch] that returns [Unit],
 * allowing it to be used as an expression in functions with a [Unit] return type.
 */
@Suppress("NOTHING_TO_INLINE")
internal inline fun CoroutineScope.launchUnit(
    context: CoroutineContext = EmptyCoroutineContext,
    start: CoroutineStart = CoroutineStart.DEFAULT,
    noinline block: suspend CoroutineScope.() -> Unit
) {
    launch(context, start, block)
}

/**
 * Kotlin mutexes are non-reentrant due to how Kotlin is designed. Reentrant mutex needs should be rare
 * however we sometimes need and there is no built in support for it. This was provided by one of the Kotlin / JetBrain leads:
 * https://elizarov.medium.com/phantom-of-the-coroutine-afc63b03a131
 */
suspend fun <T> Mutex.withReentrantLock(block: suspend () -> T): T {
    val key = ReentrantMutexContextKey(this)
    // call block directly when this mutex is already locked in the context
    if (coroutineContext[key] != null) return block()
    // otherwise add it to the context and lock the mutex
    return withContext(ReentrantMutexContextElement(key)) {
        withLock { block() }
    }
}

class ReentrantMutexContextElement(
    override val key: ReentrantMutexContextKey
) : CoroutineContext.Element

data class ReentrantMutexContextKey(
    val mutex: Mutex
) : CoroutineContext.Key<ReentrantMutexContextElement>
