package com.moloco.sdk.publisher.privacy

import androidx.preference.PreferenceManager
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.android_context.ApplicationContext

/**
 * Moloco privacy - storage for privacy related data, used in bid requests.
 */
object MolocoPrivacy {
    var privacySettings: PrivacySettings = PrivacySettings()
        private set
        get() = with(field) {
            // Updating TCF upon each property read; this way Public API isn't broken, users still can check TCF string if needed,
            // PrivacySettings instance keeps immutable, consistent, stable privacy data snapshot which then can be safely reused, compared etc.
            // If gdprApplies, use that; else legacy way
            val gdprApplies = gdprApplies()
            PrivacySettings(
                gdprApplies ?: isUserConsent,
                isAgeRestrictedUser,
                isDoNotSell,
                getTCFConsent()
            ).also {
                if (gdprApplies != null)
                    MolocoLogger.info("MolocoPrivacy", "PrivacySettings (isUserConsent/gdpr): $gdprApplies, (isAgeRestrictedUser/coppa): ${it.isAgeRestrictedUser}, (isDoNotSell/ccpa): ${it.isDoNotSell}")
                else
                    MolocoLogger.info("MolocoPrivacy", "PrivacySettings (isUserConsent/gdpr): ${it.isUserConsent}, (isAgeRestrictedUser/coppa): ${it.isAgeRestrictedUser}, (isDoNotSell/ccpa): ${it.isDoNotSell}")
                MolocoLogger.info("MolocoPrivacy", "PrivacySettings (TCF): ${it.TCFConsent}")
            }
        }

    @JvmStatic
    fun setPrivacy(privacySettings: PrivacySettings) {
        this.privacySettings = privacySettings
    }

    class PrivacySettings(
        /**
         * GDPR
         */
        val isUserConsent: Boolean? = null,
        /**
         * COPPA
         */
        val isAgeRestrictedUser: Boolean? = null,
        /**
         * CCPA
         */
        val isDoNotSell: Boolean? = null,
    ) {

        var TCFConsent: String? = null
            private set

        // Non public constructor for setting TCF string on Moloco SDK's side. For public constructor it's null by default.
        internal constructor(
            isUserConsent: Boolean?,
            isAgeRestrictedUser: Boolean?,
            isDoNotSell: Boolean?,
            tcfConsent: String?
        ) : this(isUserConsent, isAgeRestrictedUser, isDoNotSell) {
            TCFConsent = tcfConsent
        }

        // Structural equality check implementation: can't use "data class" due to TCFConsent being
        // a part of a "getter" Public API (don't want to make a breaking change)
        // and it can't be set via Public API user via constructor due to "data class" restrictions
        // and business reasons (Moloco SDK handles TCF string reads).
        override fun equals(other: Any?): Boolean {
            if (this === other) {
                return true
            }

            if (other !is PrivacySettings) {
                return false
            }

            return isUserConsent == other.isUserConsent &&
                    isAgeRestrictedUser == other.isAgeRestrictedUser &&
                    isDoNotSell == other.isDoNotSell &&
                    TCFConsent == other.TCFConsent
        }

        // Structural equality check implementation: can't use "data class" due to TCFConsent being
        // a part of Public API (don't want to make a breaking change)
        // and it can't be set via Public API user via constructor due to business reasons (Moloco SDK handles TCF string read operations).
        override fun hashCode(): Int {
            val thirtyOne = 31

            var result = isUserConsent.hashCode()
            result = thirtyOne * result + isAgeRestrictedUser.hashCode()
            result = thirtyOne * result + isDoNotSell.hashCode()
            result = thirtyOne * result + TCFConsent.hashCode()

            return result
        }

        /**
         * US_PRIVACY
         * https://github.com/InteractiveAdvertisingBureau/USPrivacy/blob/master/CCPA/US%20Privacy%20String.md
         */
        val usPrivacy: String
            get() = _usPrivacy


        // We will check if IAB US Privacy Settings are set via shared preferences first
        // If not we will use the passed isDoNotSell preferences from the Moloco Privacy API
        private val _usPrivacy: String
            get() {
                val defaultUSPrivacy = when (isDoNotSell) {
                        /**
                         * Digital property has determined the use of "us_privacy" string and CCPA does not apply to this transaction.
                         */
                        null -> "1---"
                        /**
                         * Digital property has asked a vendor to create a US Privacy String on their behalf,
                         * knowing only whether the user has opted out of sale of personal data.
                         * The user has made a choice to opt out of sale.
                         */
                        true -> "1-Y-"
                        /**
                         * Digital property has asked a vendor to create a US Privacy String on their behalf,
                         * knowing only whether the user has opted in to sale of personal data.
                         * The user has made a choice to opt in sale.
                         */
                        else -> "1-N-"
                    }

                return getUSPrivacyConsentString(defaultUSPrivacy)
            }
    }
}

/**
 * Retrieves the IAB Transparency and Consent Framework (TCF) string from the default
 * SharedPreferences using the key "IABTCF_TCString".
 *
 * @return The TCF string if it exists and is not blank, otherwise `null` is returned.
 */
private fun getTCFConsent(): String? {
    val sharedPreferences = PreferenceManager.getDefaultSharedPreferences(ApplicationContext())
    val tcfConsent = sharedPreferences.getString("IABTCF_TCString", null)

    return if (tcfConsent.isNullOrBlank()) {
        null
    } else {
        tcfConsent
    }
}

/**
 *
 * true (1) - GDPR Applies
 * false (0) - GDPR Does not apply
 * undefined (null) - unknown whether GDPR applies
 * see the below link on what does `gdprApplies` value mean
 * https://github.com/InteractiveAdvertisingBureau/GDPR-Transparency-and-Consent-Framework/blob/master/TCFv2/IAB%20Tech%20Lab%20-%20CMP%20API%20v2.md#what-does-the-gdprapplies-value-mean
 */
private fun gdprApplies(): Boolean? {
    val sharedPreferences = PreferenceManager.getDefaultSharedPreferences(ApplicationContext())
    // We need to do explicit check for the key existence because the default value can't be non-null
    if (!sharedPreferences.contains("IABTCF_gdprApplies")) {
        return null
    }

    // Is the gdprApplies a boolean or integer - https://github.com/InteractiveAdvertisingBureau/GDPR-Transparency-and-Consent-Framework/issues/340
    // > Note: For mobile all booleans are written as Number (integer)
    // https://github.com/InteractiveAdvertisingBureau/GDPR-Transparency-and-Consent-Framework/blob/master/TCFv2/IAB%20Tech%20Lab%20-%20CMP%20API%20v2.md#what-does-the-gdprapplies-value-mean
    try {
        return when (sharedPreferences.getInt("IABTCF_gdprApplies", 0)) {
            0 -> false
            1 -> true
            else -> null
        }
    } catch (e: ClassCastException) {
        // App can crash if the value is not an integer, however we can prevent a crash (and avoid being the crash reason)
        return sharedPreferences.getBoolean("IABTCF_gdprApplies", false)
    }
}

/**
 * Retrieves the US privacy consent string from the shared preferences.
 *
 * This function accesses the shared preferences using the deprecated
 * `PreferenceManager.getDefaultSharedPreferences` method to fetch the
 * value associated with the key "IABUSPrivacy_String". If the value is
 * null or blank, the function returns defaultUSPrivacy; otherwise, it returns the
 * fetched string.
 *
 * @return The US privacy consent string if available; defaultUSPrivacy otherwise.
 */
private fun getUSPrivacyConsentString(defaultUSPrivacy: String): String {
    val sharedPreferences = PreferenceManager.getDefaultSharedPreferences(ApplicationContext())
    val sharedUSPrivacy = sharedPreferences.getString("IABUSPrivacy_String", null)

    return if (sharedUSPrivacy.isNullOrBlank()) {
        defaultUSPrivacy
    } else {
        sharedUSPrivacy
    }
}
