package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.error.ErrorReportingService
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.ConnectivityService
import io.ktor.client.HttpClient
import io.ktor.client.plugins.retry
import io.ktor.client.request.get
import io.ktor.client.statement.bodyAsChannel
import io.ktor.http.contentLength
import io.ktor.util.cio.writeChannel
import io.ktor.utils.io.copyAndClose
import kotlinx.coroutines.TimeoutCancellationException
import kotlinx.coroutines.withContext
import java.io.File

@VisibleForTesting
internal const val HTTP_REQUEST_COMPLETE_TIMEOUT = "HTTP_REQUEST_COMPLETE_TIMEOUT"
internal const val HTTP_REQUEST_NOT_COMPLETE_TIMEOUT = "HTTP_REQUEST_NOT_COMPLETE_TIMEOUT"
private const val WAIT_FOR_NETWORK = 5000L


/**
 * This is the legacy implementation. Downloads the full media file at once.
 * If the file exists, the file will be removed and re-downloaded
 */
private const val MAX_HTTP_RETRIES = 10
internal class LegacyMediaDownloader(private val connectivityService: ConnectivityService,
                                     private val errorReportingService: ErrorReportingService,
                                     private val httpClient: HttpClient) : MediaDownloader {
    private val TAG = "LegacyMediaDownloader"

    override suspend fun downloadMedia(
        url: String,
        dstFile: File
    ): MediaCacheRepository.Result = withContext(DispatcherProvider().io) {
        var httpRequestComplete = false
        return@withContext try {
            MolocoLogger.info(TAG, "Fetching asset from network: $url")
            val hasNetwork = connectivityService.waitForNetwork(WAIT_FOR_NETWORK)
            if (!hasNetwork) {
                return@withContext MediaCacheRepository.Result.Failure.NoNetworkError
            }

            if (dstFile.exists()) {
                MolocoLogger.info(TAG, "Deleting existing file and re-downloading it")
                dstFile.delete()
            }

            val response = makeRequest(url)
            httpRequestComplete = true

            if (response.status.value in 400..499) {
                MolocoLogger.error(TAG, "Failed to fetch media from url: $url, status: ${response.status}")
                return@withContext MediaCacheRepository.Result.Failure.HttpClientError
            } else if (response.status.value in 500..599) {
                MolocoLogger.error(TAG, "Failed to fetch media from url: $url, status: ${response.status}")
                return@withContext MediaCacheRepository.Result.Failure.HttpServerError
            }

            val writtenBytes = response.bodyAsChannel().copyAndClose(dstFile.writeChannel())
            MolocoLogger.info(TAG, "Downloaded full response: ${response.contentLength()} and saved to disk: $writtenBytes bytes, file size: ${dstFile.length()}")
            MediaCacheRepository.Result.Success(dstFile)
        } catch (te: TimeoutCancellationException) {
            errorReportingService.reportError(timeoutErrorToSend(httpRequestComplete))
            if (httpRequestComplete) {
                MolocoLogger.error(TAG, "Timeout occurred after request had completed: $url")
            } else {
                MolocoLogger.error(TAG, "Timeout occurred when still waiting for request to complete: $url")
            }
            throw MediaFetchTimeoutException()
        } catch (e: Exception) {
            MolocoLogger.error(TAG, "Failed to fetch media from url: $url", e)
            exceptionToMediaResult(e)
        }
    }

    private suspend fun makeRequest(url: String): io.ktor.client.statement.HttpResponse {
        return httpClient.get(url) {
            retry {
                // We rely on the global timeout to do the abort for us
                // and hence we retry immediately upto 10 times before we call it quits
                maxRetries = MAX_HTTP_RETRIES
                delayMillis { _ -> 100L}
                retryOnException(MAX_HTTP_RETRIES, true)
                retryOnServerErrors(MAX_HTTP_RETRIES)
                modifyRequest {
                    MolocoLogger.info(TAG, "Retry attempt #${this.retryCount} for ${this.request.url}")
                }
            }
        }
    }

    private fun timeoutErrorToSend(hasRequestedCompleted: Boolean): String {
        return if (hasRequestedCompleted) {
            HTTP_REQUEST_COMPLETE_TIMEOUT
        } else {
            HTTP_REQUEST_NOT_COMPLETE_TIMEOUT
        }
    }
}