package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid

import android.content.Context
import android.graphics.Rect
import android.webkit.WebView
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Destroyable
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MraidAdError
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus
import org.json.JSONObject

/**
 * Function for initializing the mraid bridge
 *
 * @param context The context in which the bridge must be run
 * @param scope The context in which the bridge must be run
 * @param iframeEnabled Tells the bridge if the mraid creative should be run inside an iframe or at top level WebView.
 * When inside an iframe, we have control over the creative's audio via mediaPlaybackRequiresUserGesture flag on MraidWebview
 * @return HTML string containing an iframe with the encoded content
 */
internal fun MraidBridge(context: Context, scope: CoroutineScope, iframeEnabled: Boolean): MraidBridge =
    MraidBridgeImpl(context, scope, iframeEnabled)

// TODO. It's kind of facade now: webview apis + mraid apis. Whatever.
// Messaging mechanism WebView/mraid.js <-> SDK/Android and the rest of needed facade stuff.
internal interface MraidBridge : Destroyable {

    val mraidJsCommands: SharedFlow<MraidJsCommand>

    fun sendMraidError(command: MraidJsCommand, msg: String)
    fun sendMraidIsViewable(isViewable: Boolean)
    fun sendMraidPlacementType(placementType: MraidPlacementType)
    fun sendMraidViewState(state: MraidViewState)
    fun sendMraidSupports(
        sms: Boolean,
        tel: Boolean,
        calendar: Boolean,
        storePicture: Boolean,
        inlineVideo: Boolean,
    )

    fun sendMraidScreenMetrics(screenMetrics: MraidScreenMetrics)
    fun sendMraidReady()

    suspend fun loadMraidHtml(mraidHtml: String): Result<MraidAdData, MraidAdError>

    val isMraidHtmlPageLoaded: StateFlow<Boolean>
    val unrecoverableError: StateFlow<MraidAdError?>

    val webView: WebView
}

// TODO. Refactor.
internal interface MraidJsCommandUrlSource {
    fun consumeMraidJsCommand(fromUrl: String): Boolean
}

/**
 * Function for initializing the mraid bridge
 *
 * @param context The context in which the bridge must be run
 * @param scope The context in which the bridge must be run
 * @param iframeEnabled Tells the bridge if the mraid creative should be run inside an iframe or at top level WebView.
 * When inside an iframe, we have control over the creative's audio via mediaPlaybackRequiresUserGesture flag on MraidWebview
 * @return HTML string containing an iframe with the encoded content
 */
internal class MraidBridgeImpl(
    context: Context,
    scope: CoroutineScope,
    private val iframeEnabled: Boolean
) : MraidBridge {

    private val scope = scope + DispatcherProvider().main

    private val _mraidJsCommands = MutableSharedFlow<MraidJsCommand>()
    override val mraidJsCommands: SharedFlow<MraidJsCommand> = _mraidJsCommands

    // Must be created on main thread
    // This means the construction of the mraid bridge has side-effects, which is a code smell
    // and should be refactored
    private val _webView = MraidWebView(
        context,
        object : MraidJsCommandUrlSource {
            override fun consumeMraidJsCommand(fromUrl: String): Boolean =
                this@MraidBridgeImpl.consumeMraidJsCommand(fromUrl)
        }
    )
    override val webView: WebView = _webView

    private fun consumeMraidJsCommand(url: String): Boolean =
        when (val result = MraidJsCommand.from(url)) {
            is Result.Success -> {
                scope.launch {
                    val cmd = result.value
                    _mraidJsCommands.emit(cmd)
                    sendMraidNativeCommandComplete(cmd)
                }
                true
            }
            is Result.Failure -> {
                result.value.isMraidScheme
            }
        }

    private fun sendJs(js: String) {
        _webView.loadUrl("javascript:$js")
    }

    private fun sendMraidNativeCommandComplete(command: MraidJsCommand) {
        val cmd = JSONObject.quote(command.commandString)
        sendJs("mraidbridge.nativeCallComplete($cmd)")
    }

    override fun sendMraidReady() {
        sendJs("mraidbridge.notifyReadyEvent()")
    }

    override fun sendMraidError(command: MraidJsCommand, msg: String) {
        sendJs(
            "mraidbridge.notifyErrorEvent(${JSONObject.quote(
                command.commandString
            )}, ${JSONObject.quote(msg)})"
        )
    }

    override fun sendMraidIsViewable(isViewable: Boolean) {
        sendJs("mraidbridge.setIsViewable($isViewable)")
    }

    override fun sendMraidPlacementType(placementType: MraidPlacementType) {
        sendJs("mraidbridge.setPlacementType(${JSONObject.quote(placementType.value)})")
    }

    override fun sendMraidViewState(state: MraidViewState) {
        sendJs("mraidbridge.setState(${JSONObject.quote(state.value)})")
    }

    override fun sendMraidSupports(
        sms: Boolean,
        tel: Boolean,
        calendar: Boolean,
        storePicture: Boolean,
        inlineVideo: Boolean,
    ) {
        sendJs("mraidbridge.setSupports($sms,$tel,$calendar,$storePicture,$inlineVideo)")
    }

    override fun sendMraidScreenMetrics(screenMetrics: MraidScreenMetrics) {
        sendJs(
            """
                mraidbridge.setScreenSize(${stringifySize(screenMetrics.getScreenRectDips())});
                mraidbridge.setMaxSize(${stringifySize(screenMetrics.getRootViewRectDips())});
                mraidbridge.setCurrentPosition(${stringifyRect(
                screenMetrics.getCurrentAdRectDips()
            )});
                mraidbridge.setDefaultPosition(${stringifyRect(
                screenMetrics.getDefaultAdRectDips()
            )})
            """
        )

        // TODO. Why calling separately?
        sendJs(
            "mraidbridge.notifySizeChangeEvent(${stringifySize(
                screenMetrics.getCurrentAdRectDips()
            )})"
        )
    }

    // TODO. Refactor all of this.
    private fun stringifyRect(rect: Rect): String {
        return "${rect.left},${rect.top},${rect.width()},${rect.height()}"
    }

    // TODO. Refactor all of this.
    private fun stringifySize(rect: Rect): String {
        return "${rect.width()},${rect.height()}"
    }

    override suspend fun loadMraidHtml(mraidHtml: String): Result<MraidAdData, MraidAdError> = _webView.loadHtml(mraidHtml, iframeEnabled)

    override val isMraidHtmlPageLoaded: StateFlow<Boolean> =
        _webView.isLoaded

    override val unrecoverableError = _webView.unrecoverableError

    // It's currently implied that scope.cancel() happens outside.
    override fun destroy() {
        _webView.destroy()
    }
}
