package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.ad

import android.content.Context
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.TemplateAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.TemplateAdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.AdLoadTimeoutError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.ad.loader.WebViewAdLoad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.bridge.TemplateBridge
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.bridge.TemplateBridgeImpl
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.creative.mraid.MraidCommunicationHub
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.creative.mraid.MraidCommunicationHubImpl
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.TemplateJavascript
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.TemplateWebView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.EventHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.ClickthroughEventHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.CompositeEventHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.PlayListItemDisplayingEventHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.RequiredContentEventHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlin.time.Duration

/**
 * WebviewAd is a reusable ad object that can be used in fullscreen
 * ads or banner ads
 */
internal class WebviewAd(
    context: Context,
    adm: String,
    private val eventHandlers: Set<EventHandler>,
    private val clickthroughEventHandler: ClickthroughEventHandler,
    private val contentLoadedHandler: RequiredContentEventHandler,
    private val playListItemDisplayingEventHandler: PlayListItemDisplayingEventHandler,
): TemplateAd<TemplateAdShowListener> {
    private val scope: CoroutineScope = CoroutineScope(DispatcherProvider().main)
    private val eventHandler = CompositeEventHandler(eventHandlers)

    internal var mraidCommunicationHub : MraidCommunicationHub
    internal val webView: TemplateWebView = TemplateWebView(context, contentLoadedHandler, playListItemDisplayingEventHandler).apply {
        addJavascriptInterface(TemplateJavascript(eventHandler), "AndroidTemplateBridge")
    }.also {
        mraidCommunicationHub = MraidCommunicationHubImpl(context, it, clickthroughEventHandler)
        mraidCommunicationHub.attach()
    }


    internal val webviewBridge: TemplateBridge = TemplateBridgeImpl(webView)

    private val adLoader by lazy { WebViewAdLoad(adm, scope, webView) }

    private val _isAdDisplaying = MutableStateFlow(false)
    override val isAdDisplaying: StateFlow<Boolean> by lazy {
        // Added WebView's "isPageFinished" event check for static ads to prevent impression discrepancies,
        // between notifnt and html embedded trackers.
        _isAdDisplaying.combine(webView.isPageFinished) { isAdDisplaying, isPageFinished ->
            isAdDisplaying && isPageFinished
        }.stateIn(scope, SharingStarted.Eagerly, false)
    }

    override fun show(listener: TemplateAdShowListener) {
        scope.launch {
            _isAdDisplaying.value = true
            val error = webView.unrecoverableError.first { it != null }
            error?.let {
                listener.onShowError(it)
            }
        }
    }

    override fun destroy() {
        _isAdDisplaying.value = false
        webView.destroy()
        scope.cancel()
    }

    override val isLoaded by adLoader::isLoaded
    override fun load(timeout: Duration, listener: AdLoad.Listener?) {
        val adLoadDelegateListener = object : AdLoad.Listener {
            override fun onLoad() {
                // only send mraid events on successful load
                mraidCommunicationHub.sendMraidLoadEvents()
                listener?.onLoad()
            }

            override fun onLoadTimeout(timeoutError: AdLoadTimeoutError) {
                listener?.onLoadTimeout(timeoutError)
            }

            override fun onLoadError(internalError: MolocoAdSubErrorType) {
                listener?.onLoadError(internalError)
            }
        }

        adLoader.load(timeout, adLoadDelegateListener)
    }
}
