package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.creative.mraid

import android.content.Context
import android.webkit.WebView
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidJsCommand.SetOrientationProperties
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidOrientation
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidPlacementType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidViewState
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidViewVisualMetricsTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.ClickthroughEventHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import org.json.JSONObject

/**
 * Interface defining the communication contract between the Android SDK and the MRAID JavaScript running in a WebView.
 */
internal interface MraidCommunicationHub {
    /**
     * Attaches the communication bridge to the WebView.
     */
    fun attach()

    /**
     * Sends initial MRAID state to the creative.
     */
    fun sendMraidLoadEvents()

    /**
     * Handles the `open` MRAID command by triggering a clickthrough event with the given URL.
     *
     * @param url The destination URL to be opened.
     */
    fun handleMRAIDJsCommandOpen(url: String)

    /**
     * Handles the `close` MRAID command. Updates the view state accordingly.
     */
    fun handleMraidJsCommandClose()

    /**
     * Handles the `setOrientationProperties` MRAID command, updating the expected orientation state.
     *
     * @param allowOrientationChange Whether the creative allows orientation changes.
     * @param forceOrientation The desired orientation (portrait, landscape, or none).
     */
    fun handleMraidJsCommandSetOrientationProperties(allowOrientationChange: Boolean, forceOrientation: MraidOrientation)

    /**
     * Handles the `expand` MRAID command by updating the view state to expanded.
     */
    fun handleMraidJsCommandExpand()

    /**
     * A state flow representing the latest orientation properties as requested by MRAID.
     */
    val expectedOrientation: StateFlow<SetOrientationProperties>
}

/**
 * Implementation of [MraidCommunicationHub] that facilitates communication between Android and MRAID JavaScript.
 *
 * @property context Android context.
 * @property webView The WebView displaying the MRAID creative.
 * @property clickthroughEventHandler Handler for processing ad clickthrough events.
 */
internal class MraidCommunicationHubImpl(
    private val context: Context,
    private val webView: WebView,
    private val clickthroughEventHandler: ClickthroughEventHandler
) : MraidCommunicationHub {

    private val scope = CoroutineScope(DispatcherProvider().main)

    // Communication Bridges
    private val mraidJsEventSender = MraidJsEventSenderImpl(webView)
    private val mraidJsEventReceiver = MraidJsEventReceiver(context, this)

    // Mraid State Management
    private var mraidViewState: MraidViewState = MraidViewState.Default
    private val mraidViewVisualMetricsTracker = MraidViewVisualMetricsTracker(webView, context, scope)
    private val _expectedOrientation = MutableStateFlow(SetOrientationProperties(true, MraidOrientation.Portrait))
    override val expectedOrientation: StateFlow<SetOrientationProperties> = _expectedOrientation

    /**
     * Attaches the communication bridge by injecting the JS interface and sending initial state.
     */
    override fun attach() {
        // Listen to events from mraid creative
        webView.addJavascriptInterface(mraidJsEventReceiver, "AndroidMraid")
    }

    override fun sendMraidLoadEvents() {
        mraidViewState = MraidViewState.Default
        mraidJsEventSender.sendMraidSupports(
            sms = false,
            tel = false,
            calendar = false,
            storePicture = false,
            inlineVideo = true
        )
        mraidJsEventSender.sendMraidViewState(mraidViewState)
        mraidJsEventSender.sendMraidPlacementType(MraidPlacementType.Interstitial)
        mraidJsEventSender.sendMraidScreenMetrics(mraidViewVisualMetricsTracker.screenMetricsEvent.value.value)

        startListeningToVisualMetricsChanges()
    }

    /**
     * Handles the `open` command and emits a clickthrough event.
     */
    override fun handleMRAIDJsCommandOpen(url: String) {
        val jsonObject = JSONObject()
        val httpUrl = replaceITMSApps(url)
        jsonObject.put("event", "clickthrough")
        jsonObject.put("contentType", "mraid")
        jsonObject.put("url", httpUrl)

        try {
            clickthroughEventHandler.handleEvent(jsonObject)
            mraidJsEventSender.sendMraidNativeCommandComplete("open")
            MolocoLogger.debug("TAG", "Ad clicked: $httpUrl")
        } catch (e: Exception) {
            mraidJsEventSender.sendMraidError("open",
                "Can't open links when mraid container is not visible to the user")
        }
    }

    private fun replaceITMSApps(url: String): String {
        return if (url.startsWith("itms-apps://")) {
            url.replace("itms-apps://", "https://")
        } else {
            url
        }
    }


    /**
     * Handles the `close` command by updating the view state if currently expanded.
     */
    override fun handleMraidJsCommandClose() {
        if (mraidViewState == MraidViewState.Expanded) {
            mraidViewState = MraidViewState.Default
        }
    }

    /**
     * Updates the expected orientation state in response to the MRAID `setOrientationProperties` command.
     */
    override fun handleMraidJsCommandSetOrientationProperties(allowOrientationChange: Boolean,
                                                              forceOrientation: MraidOrientation)  {
        scope.launch {
            _expectedOrientation.emit(SetOrientationProperties(allowOrientationChange, forceOrientation))
        }
    }

    /**
     * Updates the view state to expanded in response to the `expand` MRAID command.
     */
    override fun handleMraidJsCommandExpand() {
        mraidViewState = MraidViewState.Expanded
    }

    private fun startListeningToVisualMetricsChanges() {
        mraidViewVisualMetricsTracker.isVisible.onEach {
            mraidJsEventSender.sendMraidIsViewable(it)
        }.launchIn(scope)

        mraidViewVisualMetricsTracker.screenMetricsEvent.onEach {
            mraidJsEventSender.sendMraidScreenMetrics(it.value)
        }.launchIn(scope)
    }
}
