package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers

import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmResultTag
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.EventHandler
import org.json.JSONObject

class CompositeEventHandler(
    private val eventHandlers: Set<EventHandler>
)  {

    private val processedEventHandlers by lazy {
        eventHandlers.flatMap { it.supportedEvents }.associateWith { eventName ->
            eventHandlers.filter { it.supportedEvents.contains(eventName) }
        }
    }

    fun handleEvent(event: String) {
        val timerEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.WebviewEventHandled.eventName)
        val eventPayload = JSONObject(event)
        val eventName = eventPayload.getString("event")

        MolocoLogger.debug(TAG, "Event received: $eventName")

        val handlers = processedEventHandlers[eventName]
        if (handlers.isNullOrEmpty()) {
            MolocoLogger.error(TAG, "Event not handled: $eventName")
            AndroidClientMetrics.recordCountEvent(
                CountEvent(AcmCount.WebviewEventHandled.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, "no_handler")
                    .withTag("event", eventName)
            )
            AndroidClientMetrics.recordTimerEvent(timerEvent
                .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                .withTag(AcmTag.Reason.tagName, "no_handler")
                .withTag("event", eventName))
            return
        }
        if (handlers.size > 1) {
            MolocoLogger.warn(TAG, "Found multiple event handlers for event: $eventName, using first one")
            AndroidClientMetrics.recordCountEvent(
                CountEvent(AcmCount.WebviewEventMultipleHandlers.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag("event", eventName)
            )
        }

        val eventHandlerLatency = AndroidClientMetrics.startTimerEvent(AcmTimer.WebviewEventHandlerLatency.eventName)
        val handler = handlers.first()
        try {
            handler.handleEvent(eventPayload)
            MolocoLogger.debug(TAG, "Event handled: $eventName")
            AndroidClientMetrics.recordTimerEvent(eventHandlerLatency
                .withTag("event", eventName)
                .withTag("handler", handler.TAG)
                .withTag(AcmTag.Result.tagName, AcmResultTag.success.name))
            AndroidClientMetrics.recordCountEvent(
                CountEvent(AcmCount.WebviewEventHandled.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.success.name)
                    .withTag("handler", handler.TAG)
                    .withTag("event", eventName))
            AndroidClientMetrics.recordTimerEvent(timerEvent
                .withTag(AcmTag.Result.tagName, AcmResultTag.success.name)
                .withTag("handler", handler.TAG)
                .withTag("event", eventName))
        } catch (e: Exception) {
            MolocoLogger.warn(TAG, "Event handling failed: $eventName", exception = e)
            AndroidClientMetrics.recordTimerEvent(eventHandlerLatency
                .withTag("event", eventName)
                .withTag("handler", handler.TAG)
                .withTag(AcmTag.Reason.tagName, e.javaClass.simpleName)
                .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name))
            AndroidClientMetrics.recordCountEvent(
                CountEvent(AcmCount.WebviewEventHandled.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, e.javaClass.simpleName)
                    .withTag("handler", handler.TAG)
                    .withTag("event", eventName))
            AndroidClientMetrics.recordTimerEvent(timerEvent
                .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                .withTag(AcmTag.Reason.tagName, e.javaClass.simpleName)
                .withTag("handler", handler.TAG)
                .withTag("event", eventName))
        }

    }

    companion object {
        private const val TAG = "CompositeEventHandler"
    }
}
