package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.fullscreen

import android.content.Context
import android.content.Intent
import android.content.Intent.FLAG_ACTIVITY_NEW_TASK
import android.content.pm.ActivityInfo
import android.os.Bundle
import android.view.ViewGroup
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.findViewTreeLifecycleOwner
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.hideSystemUI
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidJsCommand
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidOrientation
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.ad.WebviewAd
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import java.lang.ref.WeakReference

internal class FullscreenWebviewActivity : AppCompatActivity() {

    private fun startListeningToOrientationEvent(
        expectedOrientation: StateFlow<MraidJsCommand.SetOrientationProperties?>,
        scope: CoroutineScope = CoroutineScope(DispatcherProvider().main)
    ) {
        setOrientation(expectedOrientation.value)
        expectedOrientation.onEach(::setOrientation).launchIn(scope)
    }

    private fun setOrientation(orientationCommand: MraidJsCommand.SetOrientationProperties?) {
        orientationCommand
            ?.forceOrientation
            ?.toActivityOrientation()
            ?.let { requestedOrientation = it }
    }

    private fun MraidOrientation.toActivityOrientation() = when (this) {
        MraidOrientation.Portrait -> ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
        MraidOrientation.Landscape -> ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE
        MraidOrientation.None -> null
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        currentActivity = WeakReference(this)
        val webviewAd = webviewAd.get()
        if (webviewAd == null) {
            MolocoLogger.error(TAG, "WebviewAd is null, something went wrong")
            finish() // Close the activity if the WebView is not initialized
            return
        }

        val webView = webviewAd.webView
        if (webView.parent != null) {
            (webView.parent as ViewGroup).removeView(webView)
        }

        setContentView(webView)
        isAdDisplayingEvent?.value = true
        subscribeWebviewBridgeToLifecycleEvents(webviewAd)
        hideSystemUI()

        startListeningToOrientationEvent(webviewAd.mraidCommunicationHub.expectedOrientation)
    }

    /**
     * Subscribes the webview bridge to lifecycle events of the activity.
     */
    private fun subscribeWebviewBridgeToLifecycleEvents(webviewAd: WebviewAd) {
        val webView = webviewAd.webView
        webView.let { view ->
            webviewAd.webviewBridge.viewReady()
            val lifecycleOwner = view.findViewTreeLifecycleOwner()
            val lifecycle = lifecycleOwner?.lifecycle
            lifecycle?.addObserver(object :
                DefaultLifecycleObserver {
                override fun onPause(owner: LifecycleOwner) {
                    super.onPause(owner)
                    this@FullscreenWebviewActivity.apply {
                        // The viewVisible call can happen after the activity is destroyed
                        // on a destroyed webview, so prevent that from happening
                        if (!this.isFinishing && !this.isDestroyed) {
                            webviewAd.webviewBridge.viewVisible(false)
                        }
                    }

                }

                override fun onResume(owner: LifecycleOwner) {
                    super.onResume(owner)
                    webviewAd.webviewBridge.viewVisible(true)
                }
            })
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        isAdForciblyClosed?.value = true
        isAdForciblyClosed = null
        close()
    }

    companion object {
        private val TAG = "FullscreenWebviewActivity"

        private var webviewAd = WeakReference<WebviewAd>(null)
        private var currentActivity = WeakReference<FullscreenWebviewActivity>(null)
        private var isAdDisplayingEvent: MutableStateFlow<Boolean>? = null
        private var isAdForciblyClosed: MutableStateFlow<Boolean>? = null

        fun show(context: Context, isAdDisplayingEvent: MutableStateFlow<Boolean>, webviewAd: WebviewAd, isAdForciblyClosed: MutableStateFlow<Boolean>) {
            Companion.webviewAd = WeakReference(webviewAd)
            Companion.isAdDisplayingEvent = isAdDisplayingEvent
            Companion.isAdForciblyClosed = isAdForciblyClosed
            val intent = Intent(context, FullscreenWebviewActivity::class.java).apply {
                flags = FLAG_ACTIVITY_NEW_TASK
            }
            context.startActivity(intent)
        }

        fun close() {
            webviewAd.get()?.destroy()
            webviewAd.clear()
            webviewAd = WeakReference(null)
            isAdDisplayingEvent?.value = false
            isAdDisplayingEvent = null
            currentActivity.get()?.let { activity ->
                if (!activity.isFinishing && !activity.isDestroyed) {
                    activity.finish()
                }
            }
            currentActivity.clear()
        }
    }
}
