package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui

import android.content.Context
import android.graphics.Color
import android.view.MotionEvent
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import com.moloco.sdk.internal.SdkEventUrlTracker
import com.moloco.sdk.internal.services.ClickthroughService
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer.model.AdTouch
import kotlinx.coroutines.flow.MutableSharedFlow

/**
 * A view that intercepts touch events specifically for click handling in banners.
 * This transparent view is designed to overlay content and process touch events
 * for tracking and clickthrough.
 *
 * @property context The context in which this view is operating
 */
internal class TouchInterceptClickHandlerView(context: Context) : View(context) {
    init {
        layoutParams = FrameLayout.LayoutParams(
            ViewGroup.LayoutParams.MATCH_PARENT,
            ViewGroup.LayoutParams.MATCH_PARENT
        )
        setBackgroundColor(Color.TRANSPARENT)
    }

    /**
     * Intercepts a touch event to track clicks and potentially trigger clickthrough actions.
     * This method is specifically designed for banner click tracking and handling.
     *
     * @param event The motion event being intercepted
     * @param clickthroughService Service responsible for handling banner clickthrough events
     * @param experimentClickTrackingUrl Optional URL for tracking the click event
     * @param experimentClickthroughUrl Optional URL for redirecting the user on click
     * @param buttonTracker Tracker for button click events
     * @param clickthroughEvent Optional flow to callback when clickthrough occurs
     * @param sdkEventUrlTracker Tracker for SDK event URLs, defaults to a new instance
     */
    suspend fun interceptTouchEvent(
                            event: MotionEvent,
                            clickthroughService: ClickthroughService,
                            experimentClickTrackingUrl: String?,
                            experimentClickthroughUrl: String?,
                            buttonTracker:ButtonTracker,
                            clickthroughEvent: MutableSharedFlow<Unit>?,
                            sdkEventUrlTracker: SdkEventUrlTracker = SdkEventUrlTracker()
    ) {
        if (event.action == MotionEvent.ACTION_DOWN) {
            val position = IntArray(2)
            this.getLocationOnScreen(position)
            val lastTouch = AdTouch(
                viewPositionX = position.first(),
                viewPositionY = position.last(),
                viewSizeHeight = this.height,
                viewSizeWidth = this.width,
                touchX = (event.x + position.first()).toInt(),
                touchY = (event.y + position.last()).toInt()
            )

            experimentClickTrackingUrl?.let { trackingUrl -> sdkEventUrlTracker.invoke(url = trackingUrl, System.currentTimeMillis()) }
            experimentClickthroughUrl?.let { clickthroughUrl ->
                clickthroughService.runClickThrough(clickthroughUrl, lastTouch, buttonTracker, clickthroughEvent)
            }
        }
    }
}
