package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.platform.LocalLifecycleOwner
import androidx.compose.ui.tooling.preview.Preview
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.theme.Theme
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.ScreenUtils.toPosition
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PreparedVastResource
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.companion.CompanionViewModel
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.touch.detectTapUnconsumed
import kotlinx.coroutines.withContext

@Composable
internal fun Companion(
    viewModel: CompanionViewModel,
    overrideImageCompanionOnClick: (() -> Unit)?,
    modifier: Modifier = Modifier
) {
    LaunchedEffect(Unit) {
        withContext(DispatcherProvider().main) {
            viewModel.onDisplayStarted()
        }
    }

    DisposableEffect(Unit) {
        onDispose {
            viewModel.onDisplayEnded()
        }
    }

    val resource by viewModel.resource.collectAsState()

    val safeResource = resource
    when (safeResource) {
        is PreparedVastResource.Html -> {
            VastResourceHtml(
                modifier = modifier.pointerInput(Unit) {
                    detectTapUnconsumed { firstTapTouch, _ ->
                        viewModel.onLastClickPosition(firstTapTouch.toPosition())
                    }
                },
                htmlResource = safeResource
            )
        }
        is PreparedVastResource.Image -> CompanionVastResourceImage(
            modifier = modifier.pointerInput(Unit) {
                detectTapUnconsumed { firstTapTouch, _ ->
                    viewModel.onLastClickPosition(firstTapTouch.toPosition())
                    overrideImageCompanionOnClick?.let { it1 -> it1() } ?: viewModel.onClickThrough(firstTapTouch.toPosition())
                }
            },
            imageResource = safeResource
        )
        null -> {}
    }
}

@Composable
private fun CompanionVastResourceImage(
    imageResource: PreparedVastResource.Image,
    modifier: Modifier = Modifier,
) {
    Box(
        modifier = modifier,
        contentAlignment = Alignment.Center
    ) {
        VastResourceImage(imageResource)
    }
}

@Preview
@Composable
private fun CompanionVastResourceImagePreview() {
    Theme {
        CompanionVastResourceImage(
            modifier = Modifier.fillMaxSize(),
            imageResource = PreparedVastResource.Image(
                "https://images.unsplash.com/photo-1453728013993-6d66e9c9123a?ixlib=rb-1.2.1&ixid=MnwxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8&auto=format&fit=crop&w=2670&q=80",
                2670 / 10,
                1780 / 10
            )
        )
    }
}
