package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.linear

import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.ButtonTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.VastTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Offset
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.LinearProgressTracking
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.LinearTracking
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Position

class LinearVastTracker(
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private var click: List<String>?,
    creativeView: List<String>?,
    start: List<String>?,
    firstQuartile: List<String>?,
    midpoint: List<String>?,
    thirdQuartile: List<String>?,
    complete: List<String>?,
    private val mute: List<String>?,
    private val unMute: List<String>?,
    private val pause: List<String>?,
    private val resume: List<String>?,
    private val rewind: List<String>?,
    private val skip: List<String>?,
    private var closeLinear: List<String>?,
    progress: List<LinearProgressTracking>?,
    private val buttonTracker: ButtonTracker = ButtonTracker(),
    private val vastTracker: VastTracker = VastTracker()
) {
    fun trackClick(
        lastClickPosition: Position,
        contentPlayHead: Int? = null,
        assetUri: String? = null
    ) {
        click?.let {
            vastTracker.trackClick(
                it,
                error = null,
                contentPlayHead,
                assetUri,
                buttonTracker.renderedButtons(),
                customUserEventBuilderService,
                lastClickPosition
            )
            click = null
        }
    }

    fun trackDismiss(contentPlayHead: Int? = null, assetUri: String? = null) {
        closeLinear?.let {
            vastTracker.track(it, error = null, contentPlayHead, assetUri)
            closeLinear = null
        }
    }

    fun trackMute(contentPlayHead: Int? = null, assetUri: String? = null) {
        mute?.let { vastTracker.track(it, error = null, contentPlayHead, assetUri) }
    }

    fun trackUnMute(contentPlayHead: Int? = null, assetUri: String? = null) {
        unMute?.let { vastTracker.track(it, error = null, contentPlayHead, assetUri) }
    }

    fun trackPause(contentPlayHead: Int? = null, assetUri: String? = null) {
        pause?.let { vastTracker.track(it, error = null, contentPlayHead, assetUri) }
    }

    fun trackResume(contentPlayHead: Int? = null, assetUri: String? = null) {
        resume?.let { vastTracker.track(it, error = null, contentPlayHead, assetUri) }
    }

    fun trackRewind(contentPlayHead: Int? = null, assetUri: String? = null) {
        rewind?.let { vastTracker.track(it, error = null, contentPlayHead, assetUri) }
    }

    fun trackSkip(contentPlayHead: Int? = null, assetUri: String? = null) {
        skip?.let { vastTracker.track(it, error = null, contentPlayHead, assetUri) }
    }

    private val absoluteTrackers: List<AbsoluteProgressTracker>

    // To prevent read+write on different threads exceptions during saveState()
    // Don't want to use sync collections. Counter will do for now.
    private var absoluteTrackerItemsHandled = 0

    private val percentageTrackers: List<PercentageProgressTracker>

    // To prevent read+write on different threads  exceptions during saveState()
    // Don't want to use sync collections. Counter will do for now.
    private var percentageTrackerItemsHandled = 0

    init {
        val absoluteTrackers = mutableListOf<AbsoluteProgressTracker>()
        val percentageTrackers = mutableListOf<PercentageProgressTracker>()

        creativeView
            ?.map { AbsoluteProgressTracker(it, 0) }
            ?.let { absoluteTrackers.addAll(it) }

        start
            ?.map { AbsoluteProgressTracker(it, 0) }
            ?.let { absoluteTrackers.addAll(it) }

        firstQuartile
            ?.map { PercentageProgressTracker(it, 25) }
            ?.let { percentageTrackers.addAll(it) }

        midpoint
            ?.map { PercentageProgressTracker(it, 50) }
            ?.let { percentageTrackers.addAll(it) }

        thirdQuartile
            ?.map { PercentageProgressTracker(it, 75) }
            ?.let { percentageTrackers.addAll(it) }

        complete
            ?.map { PercentageProgressTracker(it, 100) }
            ?.let { percentageTrackers.addAll(it) }

        progress?.forEach { progressTracker ->
            when (progressTracker.offset) {
                is Offset.Time -> absoluteTrackers.add(
                    AbsoluteProgressTracker(
                        progressTracker.url,
                        progressTracker.offset.millis
                    )
                )
                is Offset.Percents -> percentageTrackers.add(
                    PercentageProgressTracker(
                        progressTracker.url,
                        progressTracker.offset.percents
                    )
                )
            }
        }

        this.absoluteTrackers = absoluteTrackers.apply { sortBy { it.millis } }
        this.percentageTrackers = percentageTrackers.apply { sortBy { it.percents } }
    }

    fun trackButtonRender(button: CustomUserEventBuilderService.UserInteraction.Button) {
        buttonTracker.onButtonRendered(button)
    }

    fun trackButtonUnrender(
        buttonType: CustomUserEventBuilderService.UserInteraction.Button.ButtonType
    ) {
        buttonTracker.onButtonUnRendered(buttonType)
    }

    fun trackProgress(assetUri: String? = null, positionMillis: Int, durationMillis: Int) {
        val percentageProgress = positionMillis / durationMillis.toDouble() * 100

        val percentageTrackersToHandle =
            percentageTrackers
                .subList(percentageTrackerItemsHandled, percentageTrackers.size)
                .takeWhile { it.percents <= percentageProgress }

        vastTracker.track(
            percentageTrackersToHandle.map { it.url },
            error = null,
            positionMillis,
            assetUri
        )

        percentageTrackerItemsHandled += percentageTrackersToHandle.size

        val absoluteTrackersToHandle =
            absoluteTrackers
                .subList(absoluteTrackerItemsHandled, absoluteTrackers.size)
                .takeWhile { it.millis <= positionMillis }

        vastTracker.track(
            absoluteTrackersToHandle.map { it.url },
            error = null,
            positionMillis,
            assetUri
        )

        absoluteTrackerItemsHandled += absoluteTrackersToHandle.size
    }

    companion object {
        fun from(
            linearTracking: LinearTracking,
            customUserEventBuilderService: CustomUserEventBuilderService
        ) = with(linearTracking) {
            LinearVastTracker(
                customUserEventBuilderService,
                click,
                creativeView,
                start,
                firstQuartile,
                midpoint,
                thirdQuartile,
                complete,
                mute,
                unMute,
                pause,
                resume,
                rewind,
                skip,
                closeLinear,
                progress
            )
        }
    }
}

private class AbsoluteProgressTracker(val url: String, val millis: Long)
private class PercentageProgressTracker(val url: String, val percents: Int)
