package com.moloco.sdk.internal

import android.text.TextUtils
import android.util.Log
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.BuildConfig
import com.moloco.sdk.internal.MolocoLogger.logEnabled
import kotlin.concurrent.thread

/**
 * MolocoLogger provides a centralized logging mechanism with the ability to enable or disable
 * logging globally.
 *
 * @property logEnabled Indicates whether logging is globally enabled.
 */
object MolocoLogger {
    /**
     * Indicates whether logging is globally enabled.
     * By default, logging is enabled in debug builds or when the
     * "debug.moloco.internal_logging" ADB property is set to true.
     */
    @JvmStatic
    var logEnabled: Boolean
        get() = with(configuration) {
            isDebugBuild() || logEnabledViaAdb() || logEnabledViaApi()
        }
        set(value) = configuration.setLogEnabledViaApi(value)

    private const val MOLOCO_TAG = "Moloco"

    /**
     * Used in the Demo app to display logs in its event logger.
     * Note: The listener can be fired on any thread. Consumers should act accordingly.
     */
    @JvmStatic
    fun addListener(loggerListener: LoggerListener) {
        listeners.add(loggerListener)
    }

    @JvmStatic
    fun removeListener(loggerListener: LoggerListener) {
        listeners.remove(loggerListener)
    }

    private var configuration: LogConfiguration = LogConfigurationImpl(AdbImpl())

    /**
     * List of LoggerListener instances subscribed to receive log events.
     */
    private val listeners = linkedSetOf<LoggerListener>()

    /**
     * Logs a message with the specified tag and message.
     *
     * @param tag The tag to associate with the log message. Default is [MOLOCO_TAG].
     * @param msg The message to log.
     * @param forceLogging If set to true, the message will be logged even if logging is not enabled. Default is false.
     */
    fun debug(tag: String = MOLOCO_TAG, msg: String, forceLogging: Boolean = false) {
        if (!logEnabled && !forceLogging) return

        val formattedTag = tag.prefixWithMolocoName()
        val formattedMsg = msg.prefixWithMethodName()

        Log.d(formattedTag, formattedMsg)
        fireListeners(formattedTag, formattedMsg)
    }

    /**
     * Logs a message with the specified tag and message only in debug builds.
     *
     * @param tag The tag to associate with the log message. Default is [MOLOCO_TAG].
     * @param msg The message to log.
     * @param forceLogging If set to true, the message will be logged even if logging is not enabled. Default is false.
     */
    fun debugBuildLog(tag: String = MOLOCO_TAG, msg: String, forceLogging: Boolean = false) {
        if (BuildConfig.DEBUG) {
            debug(tag, msg, forceLogging)
        }
    }

    /**
     * Logs a message with the specified tag and message.
     *
     * @param tag The tag to associate with the log message. Default is [MOLOCO_TAG].
     * @param msg The message to log.
     * @param forceLogging If set to true, the message will be logged even if logging is not enabled. Default is false.
     */
    fun info(tag: String = MOLOCO_TAG, msg: String, exception: Throwable? = null, forceLogging: Boolean = false) {
        if (!logEnabled && !forceLogging) return

        val formattedTag = tag.prefixWithMolocoName()
        val formattedMsg = msg.prefixWithMethodName()

        Log.i(formattedTag, formattedMsg, exception)
        fireListeners(formattedTag, formattedMsg)
    }

    /**
     * Logs a message with the specified tag and message.
     *
     * @param tag The tag to associate with the log message. Default is [MOLOCO_TAG].
     * @param msg The message to log.
     * @param forceLogging If set to true, the message will be logged even if logging is not enabled. Default is false.
     */
    fun warn(tag: String = MOLOCO_TAG, msg: String, exception: Throwable? = null, forceLogging: Boolean = false) {
        if (!logEnabled && !forceLogging) return

        val formattedTag = tag.prefixWithMolocoName()
        val formattedMsg = msg.prefixWithMethodName()

        Log.w(formattedTag, formattedMsg, exception)
        fireListeners(formattedTag, formattedMsg)
    }

    /**
     * Logs a message with the specified tag and message.
     *
     * @param tag The tag to associate with the log message. Default is [MOLOCO_TAG].
     * @param msg The message to log.
     * @param forceLogging If set to true, the message will be logged even if logging is not enabled. Default is false.
     */
    fun error(
        tag: String = MOLOCO_TAG,
        msg: String,
        exception: Throwable? = null,
        forceLogging: Boolean = false,
    ) {
        if (!logEnabled && !forceLogging) return

        val formattedTag = tag.prefixWithMolocoName()
        val formattedMsg = msg.prefixWithMethodName()

        Log.e(formattedTag, formattedMsg, exception)
        fireListeners(formattedTag, formattedMsg)
    }

    /**
     * For internal purposes only. Useful for quick logs. The default tag is `==tlog==`
     * and the [msg] will be prefixed with method name.
     */
    fun tlog(msg: String,  throwable: Throwable? = null) {
        val formattedMsg = msg.prefixWithMethodName()

        Log.i("==tlog==", formattedMsg, throwable)
    }

    @Suppress("UNUSED_VARIABLE")
    fun getCallingMethodName(): String {
        val stackTraceArray: Array<StackTraceElement> = Throwable().stackTrace

        val stackTraceElement = findMostRelevantStackTrace(stackTraceArray)
        val className = stackTraceElement.className
        val methodName = stackTraceElement.methodName

        val clazz = Class.forName(className)
        val clazzCanonical = clazz.canonicalName
        val clazzAnonymous = clazz.isAnonymousClass
        val clazzMethods = clazz.declaredMethods
        return stackTraceElement.methodName.let {
            if (it == "invokeSuspend") {
                /**
                 * From this string:
                 * foo.bar.RewardedMock$startRewardedVideoLogic$1
                 * creates this one:
                 * startRewardedVideoLogic
                 */
                val substring = stackTraceElement.className.removeSuffix(
                    "$1"
                ).substringAfterLast("$")

                substring
            } else {
                it
            }
        }
    }

    private fun fireListeners(tag: String, msg: String) = listeners.forEach {
        it.onLog(tag.prefixWithMolocoName(), msg)
    }

    /**
     * Prefixes [this] with [MOLOCO_TAG]. If already exists, then returns [this]
     */
    private fun String.prefixWithMolocoName() =
        if (startsWith(MOLOCO_TAG)) {
            this
        } else {
            MOLOCO_TAG + this
        }

    /**
     * Tries to attach caller method name to [this]. If unsuccessful returns [this]
     */
    private fun String.prefixWithMethodName() = try {
        "[${getCallingMethodName()}] $this"
    } catch (e: Exception) {
        this
    }

    /**
     * Returns the most relevant stack trace element by skipping over elements
     * belonging to the MolocoLogger class.
     *
     * @param stackTraceArray The array of stack trace elements to search through.
     * @return The most relevant stack trace element.
     */
    private fun findMostRelevantStackTrace(
        stackTraceArray: Array<StackTraceElement>,
    ): StackTraceElement {
        stackTraceArray.forEach { stackTraceElement ->
            if (stackTraceElement.className == MolocoLogger.javaClass.canonicalName) {
                // omit it
            } else {
                return@findMostRelevantStackTrace stackTraceElement
            }
        }

        // if no relevant element is found, return the first one
        return stackTraceArray.first()
    }

    interface LoggerListener {
        fun onLog(tag: String, msg: String)
    }

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    internal fun setConfiguration(configuration: LogConfiguration) {
        this.configuration = configuration
    }

    /**
     * Interface that hold various configurations to be used by the [MolocoLogger] util.
     */
    internal interface LogConfiguration {
        fun isDebugBuild(): Boolean

        /**
         * When a call is made to [MolocoLogger.logEnabled]
         */
        fun logEnabledViaApi(): Boolean

        /**
         * When logs are enabled via `"adb shell setprop debug.moloco.internal_logging true`
         */
        fun logEnabledViaAdb(): Boolean

        /**
         * Setter to be called when value is changed in [MolocoLogger.logEnabled]
         */
        fun setLogEnabledViaApi(isEnabled: Boolean)
    }

    private class LogConfigurationImpl(private val adb: Adb) : LogConfiguration {
        private var isLogEnabledViaApi: Boolean = false
        override fun isDebugBuild(): Boolean = BuildConfig.DEBUG

        override fun logEnabledViaApi(): Boolean = isLogEnabledViaApi

        override fun logEnabledViaAdb(): Boolean = adb.isLogsPropertyEnabled

        override fun setLogEnabledViaApi(isEnabled: Boolean) {
            isLogEnabledViaApi = isEnabled
        }
    }

    //region ADB - Android Debug Bridge
    internal interface Adb {
        /**
         * Indicates if the ADB property for enabling logs `debug.moloco.internal_logging` is set.
         * This param can be set like this in command shell:
         * "adb shell setprop debug.moloco.internal_logging true"
         */
        var isLogsPropertyEnabled: Boolean
    }

    /**
     * Implementation of the [Adb] interface.
     *
     * This class is responsible for managing the log properties using Android system properties.
     * It checks whether the 'debug.moloco.internal_logging' property is enabled and sets the
     * `isLogsPropertyEnabled` variable accordingly.
     *
     * Upon instantiation, a new thread is created to check if logging is enabled by calling
     * the `canShowLogs()` method. This is done asynchronously to avoid blocking the main thread.
     * The `isLogsPropertyEnabled` property is set based on the result of this check.
     *
     * @constructor Initializes the `AdbImpl` class and starts a new thread to determine
     * whether log properties are enabled.
     */
    private class AdbImpl : Adb {
        override var isLogsPropertyEnabled: Boolean = false

        init {
            thread {
                isLogsPropertyEnabled = canShowLogs()
            }
        }

        /**
         * Checks if we can show logs. This param can be set like this in command shell:
         * "adb shell setprop debug.moloco.internal_logging true"
         *
         * @return true if the 'debug.moloco.internal_logging=true' property is set.
         */
        private fun canShowLogs(): Boolean {
            val enableLogsProperty: String? = getSystemProperty(PROPERTY_MOLOCO_ENABLE_LOGS)
            return enableLogsProperty.toBoolean()
        }

        /**
         * By using reflection we can access the system properties which otherwise could only be be had by running this command in command shell:
         * "adb shell getprop"
         *
         * @param key the name of the property
         * @return value of the property or null
         */
        private fun getSystemProperty(key: String): String? {
            var value: String? = null
            try {
                value = Class.forName("android.os.SystemProperties")
                    .getMethod("get", String::class.java).invoke(null, key) as String
                if (TextUtils.isEmpty(value)) {
                    value = null
                }
            } catch (e: Exception) {
                // Something went wrong or we were only running unit test
                //  e.printStackTrace()
            }
            return value
        }

        companion object {
            private const val PROPERTY_MOLOCO_ENABLE_LOGS = "debug.moloco.internal_logging"
        }
    }
    //endregion
}
