package com.moloco.sdk.internal.ilrd

import android.content.Context
import androidx.lifecycle.ProcessLifecycleOwner
import com.moloco.sdk.Init.SDKInitResponse
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.service_locator.SdkObjectFactory
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.SupervisorJob
import kotlin.time.Duration.Companion.seconds

/**
 * Initializes and provides access to the Impression Level Revenue Data (ILRD) system.
 *
 * This class is responsible for:
 * - Initializing the ILRD system based on SDK configuration
 * - Providing reactive access to the ILRD service instance
 * - Coordinating interactions between the main SDK and ILRD components
 */
internal class IlrdInitializer(private val context: Context) {
    var ilrdService:IlrdService? = null
        private set

    /**
     * Initializes the ILRD system based on SDK configuration.
     * Ensures the ILRD service is only initialized once.
     *
     * @param response The SDK initialization response containing ILRD configuration
     */
    fun initialize(response: SDKInitResponse): Result<Unit> {
        // Return existing service if already initialized
        if (ilrdService != null) return Result.success(Unit)

        // Get and validate ILRD configuration
        val ilrdConfig = getIlrdConfig(response) ?: return Result.failure(Exception("No ILRD configuration found"))
        val supportedNetworksList = ilrdConfig.supportedNetworksList

        if (ilrdConfig.enabled.not()) {
            return Result.failure(Exception("Publisher has not opted into ILRD collection"))
        }

        if (supportedNetworksList.isEmpty()) {
            return Result.failure(Exception("No ILRD supported networks available"))
        }

        // Create scope and repository
        val scope = CoroutineScope(DispatcherProvider().default + SupervisorJob())
        val repository = createEventsRepository(scope, ilrdConfig, response)

        // Initialize service
        createIlrdService(scope, repository, supportedNetworksList)
        return Result.success(Unit)
    }

    /**
     * Extracts ILRD configuration from the response
     */
    private fun getIlrdConfig(response: SDKInitResponse): SDKInitResponse.ILRDConfig? {
        return if (response.hasIlrdConfig()) {
            response.ilrdConfig
        } else {
            null
        }
    }

    /**
     * Creates the ILRD events repository configured with parameters from the config.
     */
    private fun createEventsRepository(
        scope: CoroutineScope,
        ilrdConfig: SDKInitResponse.ILRDConfig,
        response: SDKInitResponse,
    ) = IlrdEventsRepository(
        scope = scope,
        url = ilrdConfig.rawImpUrl,
        persistentHttpRequest = SdkObjectFactory.Network.persistentHttpRequestSingleton,
        sessionExp = ilrdConfig.sessionExp.seconds,
        maxBatchSize = ilrdConfig.maxBatchSize,
        uploadInterval = ilrdConfig.uploadInterval.seconds,
        sessionMaxLength = ilrdConfig.maxSessionLen.seconds,
        timeProvider = SdkObjectFactory.Miscellaneous.timeProviderSingleton,
        processLifeycle = ProcessLifecycleOwner.get().lifecycle,
        advertisingIdService = SdkObjectFactory.Miscellaneous.adDataSingleton,
        pubId = response.publisherId,
        appId = response.appId,
        dataStoreService = SdkObjectFactory.Storage.dataStoreSingleton,
    )

    /**
     * Creates and initializes the ILRD service.
     */
    private fun createIlrdService(
        scope: CoroutineScope,
        repository: IlrdEventsRepository,
        supportedNetworksList: List<SDKInitResponse.SupportedNetworks>,
    ) = IlrdService(scope, context, repository, supportedNetworksList).apply {
        ilrdService = this
        subscribe()
    }

    companion object {
        private const val TAG = "IlrdInitializer"
    }
}
