package com.moloco.sdk.internal.ilrd

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.services.TimeProviderService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.text.SimpleDateFormat
import kotlin.time.Duration

/**
 * Scheduler for running delayed tasks in ILRD subsystem.
 *
 * This class provides the ability to schedule a single task with a specified delay.
 * When a new task is scheduled, any previously scheduled task is automatically cancelled.
 * This makes it suitable for implementing timeout/expiry mechanisms where only the most
 * recently scheduled timeout should be active.
 *
 * All operations are thread-safe through synchronization.
 *
 * @property scope The scope in which tasks will be launched
 * @property timeProvider Service that provides the current time
 * @property name Identifier for this scheduler instance, used in logging
 */
internal class IlrdScheduler(
    private val scope: CoroutineScope,
    private val timeProvider: TimeProviderService,
    private val name: String,
) {
    private var job: Job? = null
    private val dateFormatter = SimpleDateFormat("HH:mm:ss")

    /**
     * Schedules a task to be executed after the specified delay.
     *
     * If there is already a scheduled task, it will be cancelled before
     * scheduling the new one. This ensures that only one task is active at any time.
     *
     * The scheduling and invocation events are logged for debugging purposes.
     *
     * @param delay The duration to wait before executing the task
     * @param task The task to execute after the delay elapses
     */
    @Synchronized
    fun schedule(delay: Duration, task: suspend () -> Unit) {
        job?.let { cancelCurrentJob(it) }

        job = scope.launch {
            val formattedScheduledTime = dateFormatter.format(timeProvider.currentTime() + delay.inWholeMilliseconds)
            MolocoLogger.info(TAG, "Task $name scheduled at $formattedScheduledTime")

            delay(delay)

            // create a new scope for the task at hand, so that the existing job (scheduled task) gets cancelled.
            scope.launch {
                MolocoLogger.info(TAG, "Task $name invoked")
                task.invoke()
            }
        }
    }

    /**
     * Cancels the currently scheduled task, if one exists and is still active.
     *
     * This is called automatically when scheduling a new task, but can also
     * be used to explicitly cancel a scheduled task without scheduling a new one.
     * Cancellation is logged for debugging purposes.
     */
    @VisibleForTesting
    fun cancelCurrentJob(job: Job) = with(job) {
        if (isActive.not()) return@with

        cancel()
        MolocoLogger.info(TAG, "Task $name cancelled")
    }

    companion object {
        private const val TAG = "IlrdScheduler"
    }
}
