package com.moloco.sdk.internal.ilrd.provider

import android.content.Context
import android.os.Bundle
import com.applovin.communicator.AppLovinCommunicator
import com.applovin.communicator.AppLovinCommunicatorMessage
import com.applovin.communicator.AppLovinCommunicatorSubscriber
import com.moloco.sdk.IlrdRequest
import com.moloco.sdk.internal.ilrd.IlrdProvider
import com.moloco.sdk.internal.ilrd.IlrdState
import com.moloco.sdk.internal.ilrd.model.IlrdMediationPlatform
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch


/**
 * Handles impression level revenue data (ILRD) from AppLovin SDK.
 *
 * IMPORTANT: This class requires the AppLovin SDK to be included in the project.
 * If the AppLovin SDK is not available, the subscribe and unsubscribe operations
 * will fail gracefully with log messages.
 *
 * The class uses reflection to check for AppLovin's availability at runtime,
 * but cannot function without the actual SDK dependency being included by the
 * consuming application.
 */
internal class ApplovinIlrd(
    private val context: Context,
    private val scope: CoroutineScope,
) : IlrdProvider {
    override val mediationPlatform = IlrdMediationPlatform.MAX
    override val state by lazy { _state.asStateFlow() }
    override val events by lazy { _events.asSharedFlow() }

    private val _state = MutableStateFlow<IlrdState>(IlrdState.Unsubscribed)
    private val _events = MutableSharedFlow<IlrdProvider.IlrdImpression.Max>()

    /**
     * Subscribes to AppLovin impression level revenue events.
     *
     * This operation requires the AppLovin SDK to be available in the project.
     * If the SDK is not available, the operation will fail gracefully with a log message.
     *
     */
    @Synchronized
    override fun subscribe(): Result<Unit> {
        val result = subscribeImpl()
        result.onFailure {
            _state.value = IlrdState.Error(it.toString())
        }
        result.onSuccess {
            _state.value = IlrdState.Subscribed
        }

        return result
    }

    /**
     * Attempts to access the AppLovin Communicator instance and subscribe to revenue events.
     *
     * This method uses reflection to check if the AppLovin SDK classes are available.
     * If the AppLovin SDK is not included in the project, this will return a failure result.
     *
     */
    private fun subscribeImpl(): Result<Unit> {
        val result = runCatching {
            Class.forName("com.applovin.communicator.AppLovinCommunicator")
            Class.forName("com.applovin.communicator.AppLovinCommunicatorMessage")
            Class.forName("com.applovin.communicator.AppLovinCommunicatorSubscriber")
            AppLovinCommunicator.getInstance(context)
        }

        val communicator = result.getOrElse { return Result.failure(it) }

        // If we are here, AppLovin is added as a dependency
        createCallback().also {
            _callback = it
            communicator.subscribe(it, MAX_REVENUE_EVENTS)
        }
        return Result.success(Unit)
    }

    /**
     * IMPORTANT: Lazy instance of the AppLovin Communicator subscriber. This is lazily initialized
     * to avoid instantiation if the AppLovin SDK is not present.
     */
    private fun createCallback() = object : AppLovinCommunicatorSubscriber {
        override fun getCommunicatorId(): String = "Moloco"
        override fun onMessageReceived(message: AppLovinCommunicatorMessage) {
            // Don't waste precious resources if we are not active
            if (scope.isActive.not()) return

            // In the case that you are subscribed to multiple topics, check for the desired one
            if (MAX_REVENUE_EVENTS == message.topic) {
                val data: Bundle = message.messageData

                val model = createModel(data)


                scope.launch {
                    _events.emit(model)
                }
            }
        }
    }

    private fun createModel(data: Bundle) = run {
        val revenue = data.getDouble("revenue")
        val countryCode = data.getString("country_code")
        val networkName = data.getString("network_name")
        val adUnitId = data.getString("max_ad_unit_id")
        val thirdPartyAdPlacementId = data.getString("third_party_ad_placement_id")
        val adFormat = data.getString("ad_format")
        val userSegment = data.getString("user_segment")
        val id = data.getString("id")

        IlrdProvider.IlrdImpression.Max(IlrdRequest.MaxImpression.newBuilder()
            .apply {
                setRevenue(revenue)
                countryCode?.let { setCountryCode(it) }
                networkName?.let { setNetworkName(it) }
                adUnitId?.let { setMaxAdUnitId(it) }
                thirdPartyAdPlacementId?.let { setThirdPartyAdPlacementId(it) }
                adFormat?.let { setAdFormat(it) }
                userSegment?.let { setUserSegment(it) }
                id?.let { setId(it) }
            }
            .build()
        )
    }

    companion object {
        /**
         * Hard reference must be kept, otherwise AppLovin will garbage collect it.
         * Volatile annotation is required in order for the field to not be optimized out by the compiler
         */
        @Volatile
        private var _callback: AppLovinCommunicatorSubscriber? = null
        private const val MAX_REVENUE_EVENTS = "max_revenue_events"
        private const val TAG = "ApplovinIlrd"
    }
}
