package com.moloco.sdk.internal.ortb

import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.ortb.model.BidResponse
import com.moloco.sdk.internal.ortb.model.SeatBid

/**
 * Substitutes burl and adm fields occurencies of ${AUCTION_PRICE} macros with respective bid.price
 *
 * [JIRA](https://mlc.atlassian.net/browse/SDK-723)
 * @return copy of this@[BidResponse] with ${AUCTION_PRICE} macros substituted for adm and burl fields.
 */
internal fun BidResponse.substituteAuctionPriceMacros(): BidResponse {
    val newSeatBid = seatBid.map { sb ->
        SeatBid(
            sb.bid.map { bid ->
                val price = bid.price

                val newAdm = bid.adm.substituteAuctionPriceMacros(price)
                val newBurl = bid.burl?.substituteAuctionPriceMacros(price)

                Bid(
                    adm = newAdm,
                    burl = newBurl,
                    price = price,
                    ext = bid.ext
                )
            }
        )
    }

    return BidResponse(newSeatBid)
}

internal fun String.substituteAuctionPriceMacros(price: Float?): String {
    return replace(AUCTION_PRICE, price?.toString() ?: "")
}

private val AUCTION_PRICE = """\$\{AUCTION_PRICE\}""".toRegex()
