package com.moloco.sdk.internal.publisher

import android.annotation.SuppressLint
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.acm.TimerEvent
import com.moloco.sdk.internal.AdFactory
import com.moloco.sdk.internal.BannerSize
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.mediators.Mediators
import com.moloco.sdk.internal.mediators.bannerAdLoadTimeout
import com.moloco.sdk.internal.mediators.fullScreenAdLoadTimeout
import com.moloco.sdk.internal.mediators.nativeAdLoadTimeout
import com.moloco.sdk.internal.publisher.CreateAdType.BANNER
import com.moloco.sdk.internal.publisher.CreateAdType.BANNER_TABLET
import com.moloco.sdk.internal.publisher.CreateAdType.INTERSTITIAL
import com.moloco.sdk.internal.publisher.CreateAdType.MREC
import com.moloco.sdk.internal.publisher.CreateAdType.NATIVE_AD_MEDIATION
import com.moloco.sdk.internal.publisher.CreateAdType.REWARDED
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.TimeProviderService
import com.moloco.sdk.publisher.AdFormatType
import com.moloco.sdk.publisher.Banner
import com.moloco.sdk.publisher.Initialization
import com.moloco.sdk.publisher.InterstitialAd
import com.moloco.sdk.publisher.Moloco
import com.moloco.sdk.publisher.MolocoAdError
import com.moloco.sdk.publisher.NativeAd
import com.moloco.sdk.publisher.RewardedInterstitialAd
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.WatermarkImpl
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeoutOrNull

/**
 * The `AdCreator` class is responsible for creating various types of ads by using a suspend function
 * to await the AdFactory object, which is provided through the constructor.
 *
 * @property awaitAdFactory A suspend function that retrieves the AdFactory used to create ads.
 */
internal class AdCreator(
        private val initializationState: StateFlow<Initialization?>,
        private val timeProviderService: TimeProviderService,
        private val adCreatorConfiguration: AdCreatorConfiguration,
        private val awaitAdFactory: suspend () -> AdFactory,
) {
    /**
     * Creates a Banner ad using the provided adUnitId and optional watermark string.
     *
     * @param adUnitId The ID of the ad unit.
     * @param watermarkString Optional watermark to be included in the ad.
     * @return A `Banner` object, or `null` if the ad could not be created.
     */
    suspend fun createBanner(adUnitId: String, watermarkString: String? = null) = withContext(coroutineContext) {
        val adType = BANNER
        val createAdTime = timeProviderService()
        val isSdkInitialized = initialSdkState()
        val createTimerEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.CreateAd.eventName)
                                                        .withTag(AcmTag.AdType.tagName, adType.name)
                                                        // Tag to indicate that when the createXXX call was initiated, whether the sdk was initialized
                                                        .withTag("initial_sdk_init_state",
                                                            isSdkInitialized
                                                    )
        MolocoLogger.info(TAG, "Creating $adType ad with adUnitId: $adUnitId")
        val ad = awaitAdFactoryWithTimeoutOrNull(awaitAdFactory, adType)?.createBanner(
                SdkObjectFactory.context,
                SdkObjectFactory.Analytics.applicationLifecycleTrackerSingleton,
                adUnitId,
                SdkObjectFactory.Miscellaneous.viewVisibilityTrackerFactory,
                SdkObjectFactory.Miscellaneous.externalLinkHandlerFactory,
                WatermarkImpl(watermarkString),
                AdCreateLoadTimeoutManager(AdFormatType.BANNER, bannerAdLoadTimeout(Moloco.mediationInfo?.name)),
                SdkObjectFactory.Miscellaneous.viewLifecycleOwnerSingleton,
                BannerSize.Standard
        ) ?: kotlin.run {
            val error = handleAdNullScenario(adUnitId, isSdkInitialized, createTimerEvent, adType)
            MolocoLogger.warn(TAG, "Failed to create $adType with reason: $error")
            return@withContext Result.Failure(error)
        }

        AndroidClientMetrics.recordCountEvent(
            CountEvent(AcmCount.CreateAd.eventName)
                .withTag(AcmTag.Result.tagName, "success").withTag(AcmTag.AdType.tagName, adType.name)
                .withTag("initial_sdk_init_state", isSdkInitialized)
        )
        AndroidClientMetrics.recordTimerEvent(createTimerEvent.withTag(AcmTag.Result.tagName, "success"))

        /*
         * Not part of the public API, hence the check. Perhaps in the future we could get the implementation from the AdFactory
         * and return the public interface from this function...
         */
        if (ad is CreateAdObjectTime) {
            ad.createAdObjectStartTime = createAdTime
        }

        MolocoLogger.info(TAG, "Created $adType ad with adUnitId: $adUnitId")
        Result.Success<Banner, MolocoAdError.AdCreateError>(ad)
    }

    /**
     * Creates a Banner ad for tablet devices using the provided adUnitId and optional watermark string.
     *
     * @param adUnitId The ID of the ad unit.
     * @param watermarkString Optional watermark to be included in the ad.
     * @return A `Banner` object, or `null` if the ad could not be created.
     */
    suspend fun createBannerTablet(adUnitId: String, watermarkString: String? = null) = withContext(coroutineContext) {
        val adType = BANNER_TABLET
        val createAdTime = timeProviderService()
        val isSdkInitialized = initialSdkState()
        val createTimerEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.CreateAd.eventName)
            .withTag(AcmTag.AdType.tagName, adType.name)
            // Tag to indicate that when the createXXX call was initiated, whether the sdk was initialized
            .withTag("initial_sdk_init_state",
                isSdkInitialized
            )
        MolocoLogger.info(TAG, "Creating $adType ad with adUnitId: $adUnitId")
        val ad = awaitAdFactoryWithTimeoutOrNull(awaitAdFactory, adType)?.createBannerTablet(
                SdkObjectFactory.context,
                SdkObjectFactory.Analytics.applicationLifecycleTrackerSingleton,
                adUnitId,
                SdkObjectFactory.Miscellaneous.viewVisibilityTrackerFactory,
                SdkObjectFactory.Miscellaneous.externalLinkHandlerFactory,
                WatermarkImpl(watermarkString),
                AdCreateLoadTimeoutManager(AdFormatType.BANNER, bannerAdLoadTimeout(Moloco.mediationInfo?.name)),
                SdkObjectFactory.Miscellaneous.viewLifecycleOwnerSingleton,
                BannerSize.Tablet
        ) ?: kotlin.run {
            val error = handleAdNullScenario(adUnitId, isSdkInitialized, createTimerEvent, adType)
            MolocoLogger.warn(TAG, "Failed to create $adType with reason: $error")
            return@withContext Result.Failure(error)
        }

        AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.CreateAd.eventName).withTag(AcmTag.Result.tagName, "success").withTag(AcmTag.AdType.tagName, adType.name).withTag("initial_sdk_init_state",
            isSdkInitialized
        ))
        AndroidClientMetrics.recordTimerEvent(createTimerEvent.withTag(AcmTag.Result.tagName, "success"))

        if (ad is CreateAdObjectTime) {
            ad.createAdObjectStartTime = createAdTime
        }

        MolocoLogger.info(TAG, "Created $adType ad with adUnitId: $adUnitId")
        Result.Success<Banner, MolocoAdError.AdCreateError>(ad)
    }

    /**
     * Creates an MREC ad using the provided adUnitId and optional watermark string.
     *
     * @param adUnitId The ID of the ad unit.
     * @param watermarkString Optional watermark to be included in the ad.
     * @return A `Banner` object, or `null` if the ad could not be created.
     */
    suspend fun createMREC(adUnitId: String, watermarkString: String? = null) = withContext(coroutineContext) {
        val adType = MREC
        val createAdTime = timeProviderService.currentTime()
        val isSdkInitialized = initialSdkState()
        val createTimerEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.CreateAd.eventName).withTag(AcmTag.AdType.tagName, adType.name)
            .withTag("initial_sdk_init_state",
                isSdkInitialized
            )
        MolocoLogger.info(TAG, "Creating $adType ad with adUnitId: $adUnitId")
        val ad = awaitAdFactoryWithTimeoutOrNull(awaitAdFactory, adType)?.createMREC(
                SdkObjectFactory.context,
                SdkObjectFactory.Analytics.applicationLifecycleTrackerSingleton,
                adUnitId,
                SdkObjectFactory.Miscellaneous.viewVisibilityTrackerFactory,
                SdkObjectFactory.Miscellaneous.externalLinkHandlerFactory,
                WatermarkImpl(watermarkString),
                AdCreateLoadTimeoutManager(AdFormatType.MREC, bannerAdLoadTimeout(Moloco.mediationInfo?.name)),
                SdkObjectFactory.Miscellaneous.viewLifecycleOwnerSingleton,
                BannerSize.MREC
        ) ?: kotlin.run {
            val error = handleAdNullScenario(adUnitId, isSdkInitialized, createTimerEvent, adType)
            MolocoLogger.warn(TAG, "Failed to create $adType with reason: $error")
            return@withContext Result.Failure(error)
        }

        AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.CreateAd.eventName).withTag(AcmTag.Result.tagName, "success").withTag(AcmTag.AdType.tagName, adType.name).withTag("initial_sdk_init_state", isSdkInitialized.toString()))
        AndroidClientMetrics.recordTimerEvent(createTimerEvent.withTag(AcmTag.Result.tagName, "success"))

        if (ad is CreateAdObjectTime) {
            ad.createAdObjectStartTime = createAdTime
        }

        MolocoLogger.info(TAG, "Created $adType ad with adUnitId: $adUnitId")
        Result.Success<Banner, MolocoAdError.AdCreateError>(ad)
    }

    /**
     * Creates a Native Ad for mediation using the provided adUnitId.
     */
    suspend fun createNativeAd(adUnitId: String, watermarkString: String? = null) = withContext(coroutineContext) {
        val adType = NATIVE_AD_MEDIATION
        val createAdTime = timeProviderService.currentTime()
        val isSdkInitialized = initialSdkState()
        val createTimerEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.CreateAd.eventName).withTag(AcmTag.AdType.tagName, adType.name)
            .withTag("initial_sdk_init_state", isSdkInitialized)
        MolocoLogger.info(TAG, "Creating $adType ad with adUnitId: $adUnitId")
        val ad = awaitAdFactoryWithTimeoutOrNull(awaitAdFactory, adType)?.createNativeAd(
                SdkObjectFactory.context,
                SdkObjectFactory.Analytics.applicationLifecycleTrackerSingleton,
                SdkObjectFactory.DeviceAndApplicationInfo.audioSingleton,
                adUnitId,
                SdkObjectFactory.Miscellaneous.viewVisibilityTrackerFactory,
                SdkObjectFactory.Miscellaneous.externalLinkHandlerFactory,
                SdkObjectFactory.Network.persistentHttpRequestSingleton,
                SdkObjectFactory.Miscellaneous.viewLifecycleOwnerSingleton,
                WatermarkImpl(watermarkString),
                AdCreateLoadTimeoutManager(AdFormatType.NATIVE, nativeAdLoadTimeout(Moloco.mediationInfo?.name)),
                SdkObjectFactory.Miscellaneous.timeProviderSingleton,
                SdkObjectFactory.AdLoadModule.webViewAvailabilityCheckerSingleton,
        ) ?: kotlin.run {
            val error = handleAdNullScenario(adUnitId, isSdkInitialized, createTimerEvent, adType)
            MolocoLogger.warn(TAG, "Failed to create $adType with reason: $error")
            return@withContext Result.Failure(error)
        }

        AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.CreateAd.eventName).withTag(AcmTag.Result.tagName, "success").withTag(AcmTag.AdType.tagName, adType.name).withTag("initial_sdk_init_state", isSdkInitialized.toString()))
        AndroidClientMetrics.recordTimerEvent(createTimerEvent.withTag(AcmTag.Result.tagName, "success"))

        if (ad is CreateAdObjectTime) {
            ad.createAdObjectStartTime = createAdTime
        }

        MolocoLogger.info(TAG, "Created $adType ad with adUnitId: $adUnitId")
        Result.Success<NativeAd, MolocoAdError.AdCreateError>(ad)
    }

    /**
     * Creates an Interstitial ad using the provided adUnitId and optional watermark string.
     *
     * @param adUnitId The ID of the ad unit.
     * @param watermarkString Optional watermark to be included in the ad.
     * @return An `InterstitialAd` object, or `null` if the ad could not be created.
     */
    suspend fun createInterstitial(adUnitId: String, watermarkString: String? = null) = withContext(coroutineContext) {
        val adType = INTERSTITIAL
        val createAdTime = timeProviderService()
        val isSdkInitialized = initialSdkState()
        val createTimerEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.CreateAd.eventName)
            .withTag(AcmTag.AdType.tagName, adType.name).withTag("initial_sdk_init_state",
                isSdkInitialized
            )
        MolocoLogger.info(TAG, "Creating $adType ad with adUnitId: $adUnitId")
        val ad = awaitAdFactoryWithTimeoutOrNull(awaitAdFactory, adType)?.createInterstitial(
                SdkObjectFactory.context,
                SdkObjectFactory.Analytics.applicationLifecycleTrackerSingleton,
                adUnitId,
                SdkObjectFactory.Miscellaneous.viewVisibilityTrackerFactory,
                SdkObjectFactory.Miscellaneous.externalLinkHandlerFactory,
                SdkObjectFactory.Network.persistentHttpRequestSingleton,
                WatermarkImpl(watermarkString),
                AdCreateLoadTimeoutManager(AdFormatType.INTERSTITIAL, fullScreenAdLoadTimeout(Moloco.mediationInfo?.name))
        ) ?: kotlin.run {
            val error = handleAdNullScenario(adUnitId, isSdkInitialized, createTimerEvent, adType)
            MolocoLogger.warn(TAG, "Failed to create $adType with reason: $error")
            return@withContext Result.Failure(error)
        }

        AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.CreateAd.eventName).withTag(AcmTag.Result.tagName, "success").withTag(AcmTag.AdType.tagName, adType.name).withTag("initial_sdk_init_state", isSdkInitialized.toString()))
        AndroidClientMetrics.recordTimerEvent(createTimerEvent.withTag(AcmTag.Result.tagName, "success"))

        if (ad is CreateAdObjectTime) {
            ad.createAdObjectStartTime = createAdTime
        }

        MolocoLogger.info(TAG, "Created $adType ad with adUnitId: $adUnitId")
        Result.Success<InterstitialAd, MolocoAdError.AdCreateError>(ad)
    }

    /**
     * Creates a Rewarded Interstitial ad using the provided adUnitId and optional watermark string.
     *
     * @param adUnitId The ID of the ad unit.
     * @param watermarkString Optional watermark to be included in the ad.
     * @return A `RewardedInterstitialAd` object, or `null` if the ad could not be created.
     */
    suspend fun createRewardedInterstitial(adUnitId: String, watermarkString: String? = null) = withContext(coroutineContext) {
        val adType = REWARDED
        val createAdTime = timeProviderService()
        val isSdkInitialized = initialSdkState()
        val createTimerEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.CreateAd.eventName).withTag(AcmTag.AdType.tagName, adType.name)
            .withTag("initial_sdk_init_state", isSdkInitialized)
        MolocoLogger.info(TAG, "Creating $adType ad with adUnitId: $adUnitId")
        val ad = awaitAdFactoryWithTimeoutOrNull(awaitAdFactory, adType)?.createRewardedInterstitial(
                SdkObjectFactory.context,
                SdkObjectFactory.Analytics.applicationLifecycleTrackerSingleton,
                adUnitId,
                SdkObjectFactory.Miscellaneous.viewVisibilityTrackerFactory,
                SdkObjectFactory.Miscellaneous.externalLinkHandlerFactory,
                SdkObjectFactory.Network.persistentHttpRequestSingleton,
                WatermarkImpl(watermarkString),
                AdCreateLoadTimeoutManager(AdFormatType.REWARDED, fullScreenAdLoadTimeout(Moloco.mediationInfo?.name))
        ) ?: kotlin.run {
            val error = handleAdNullScenario(adUnitId, isSdkInitialized, createTimerEvent, adType)
            MolocoLogger.warn(TAG, "Failed to create $adType with reason: $error")
            return@withContext Result.Failure(error)
        }

        AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.CreateAd.eventName).withTag(AcmTag.Result.tagName, "success").withTag(AcmTag.AdType.tagName, adType.name).withTag("initial_sdk_init_state",
            isSdkInitialized
        ))
        AndroidClientMetrics.recordTimerEvent(createTimerEvent.withTag(AcmTag.Result.tagName, "success"))

        if (ad is CreateAdObjectTime) {
            ad.createAdObjectStartTime = createAdTime
        }

        MolocoLogger.info(TAG, "Created $adType ad with adUnitId: $adUnitId")
        Result.Success<RewardedInterstitialAd, MolocoAdError.AdCreateError>(ad)
    }

    private fun handleAdNullScenario(adUnitId: String, initialSdkInitializationState: String, createTimerEvent: TimerEvent, adType: CreateAdType): MolocoAdError.AdCreateError {
        val createEvent = CountEvent(AcmCount.CreateAd.eventName)
                .withTag(AcmTag.Result.tagName, "failure")
                .withTag("initial_sdk_init_state", initialSdkInitializationState)
                .withTag(AcmTag.AdType.tagName, adType.name)
        return when (initializationState.value) {
            Initialization.SUCCESS -> {
                // This scenario can only happen when sdk init completed successfully right when the timeout happen so ad creation failed
                val errorMsg = UNABLE_TO_CREATE_AD
                SdkObjectFactory.Analytics.errorReportingSingleton.reportError(errorMsg)
                AndroidClientMetrics.recordTimerEvent(createTimerEvent.withTag(AcmTag.Result.tagName, "failure").withTag(AcmTag.Reason.tagName, "unable_to_create_ad"))
                AndroidClientMetrics.recordCountEvent(
                        createEvent.withTag(AcmTag.Reason.tagName, "unable_to_create_ad")
                )
                MolocoLogger.error(
                        TAG,
                        "Could not find the adUnitId that was requested for load: $adUnitId"
                )
                MolocoAdError.AdCreateError.UNABLE_TO_CREATE_AD
            }

            Initialization.FAILURE -> {
                SdkObjectFactory.Analytics.errorReportingSingleton.reportError("CREATE_${adType.name.uppercase()}_AD_FAILED_SDK_INIT_FAILED")
                AndroidClientMetrics.recordTimerEvent(createTimerEvent.withTag(AcmTag.Result.tagName, "failure").withTag(AcmTag.Reason.tagName, "sdk_init_failed"))
                AndroidClientMetrics.recordCountEvent(
                        createEvent.withTag(AcmTag.Reason.tagName, "sdk_init_failed")
                )
                MolocoLogger.error(TAG, "Cannot create AdFactory as SDK init was failure")
                MolocoAdError.AdCreateError.SDK_INIT_FAILED
            }

            null -> {
                SdkObjectFactory.Analytics.errorReportingSingleton.reportError("CREATE_${adType.name.uppercase()}_AD_FAILED_SDK_INIT_NOT_COMPLETED")
                AndroidClientMetrics.recordTimerEvent(createTimerEvent.withTag(AcmTag.Result.tagName, "failure").withTag(AcmTag.Reason.tagName, "sdk_init_not_completed"))
                AndroidClientMetrics.recordCountEvent(
                        createEvent.withTag(AcmTag.Reason.tagName, "sdk_init_not_completed")
                )
                MolocoLogger.error(TAG, "Cannot retrieve AdFactory as SDK init was not called or not completed")
                MolocoAdError.AdCreateError.SDK_INIT_WAS_NOT_COMPLETED
            }
        }
    }

    /**
     * Awaits the result of the `awaitAdFactory` function with a timeout based on the provided ad format type.
     * If the operation exceeds the allotted timeout duration, the function returns `null`.
     *
     * @param awaitAdFactory A suspending function that provides an instance of [AdFactory].
     * @param createAdType The type of ad for which the timeout duration is determined.
     * @return An instance of [AdFactory] if it completes within the timeout duration, or `null` if the operation times out.
     *
     * Timeout durations:
     * - BANNER and MREC: 5 seconds.
     * - INTERSTITIAL and REWARDED: 15 seconds.
     * - NATIVE_SMALL_IMAGE, NATIVE_MEDIUM_IMAGE, NATIVE_MEDIUM_VIDEO: 15 seconds.
     */
    @SuppressLint("RestrictedApi")
    private suspend fun awaitAdFactoryWithTimeoutOrNull(
        awaitAdFactory: suspend () -> AdFactory,
        createAdType: CreateAdType,
    ): AdFactory? {
        val awaitTimer = AndroidClientMetrics.startTimerEvent(AcmTimer.CreateAdAwaitAdFactory.eventName)

        /**
         * Ideally, the timeouts should be set to 50% of the values defined in [Mediators].
         * However, the current implementation of [Mediators] already uses reduced timeout values.
         * Once the original (non-reduced) timeout values are included in [Mediators],
         * the timeout for this function can be dynamically retrieved from there.
         */
        val awaitAdFactoryTimeout = adCreatorConfiguration.adTimeouts[createAdType] ?: adCreatorConfiguration.defaultTimeoutDuration

        MolocoLogger.info(TAG, "Waiting for AdFactory with timeout: $awaitAdFactoryTimeout")
        return withTimeoutOrNull(awaitAdFactoryTimeout) { awaitAdFactory() }.also { adFactory ->
            MolocoLogger.info(TAG, "AdFactory received: ${adFactory != null}")
                AndroidClientMetrics.recordTimerEvent(
                    awaitTimer.withTag(AcmTag.AdType.tagName, createAdType.name)
                        .withTag(AcmTag.Result.tagName, if (adFactory != null) "success" else "failure")
                )
            }
    }

    private val coroutineContext = DispatcherProvider().default



    private fun initialSdkState(): String = initializationState.value?.name?.lowercase() ?: "not_invoked_or_in_progress"

    companion object {
        internal const val UNABLE_TO_CREATE_AD = "UNABLE_TO_CREATE_AD"

        private const val TAG = "AdCreator"
    }
}
