package com.moloco.sdk.internal.publisher.nativead

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.acm.TimerEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.publisher.AdCreateLoadTimeoutManager
import com.moloco.sdk.internal.publisher.AdLoadListenerTracker
import com.moloco.sdk.internal.publisher.CreateAdObjectTime
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.AnalyticsApplicationLifecycleTracker
import com.moloco.sdk.publisher.AdFormatType
import com.moloco.sdk.publisher.AdLoad
import com.moloco.sdk.publisher.Moloco
import com.moloco.sdk.publisher.NativeAd
import com.moloco.sdk.publisher.createAdInfo
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancel
import kotlinx.coroutines.launch
import com.moloco.sdk.acm.AndroidClientMetrics as acm

/**
 * Implementation of the [NativeAd] interface for managing and displaying native advertisements.
 *
 * This class is responsible for handling the lifecycle of a native ad, including loading, rendering,
 * impression tracking, and click tracking. It uses a combination of dependency services, utilities,
 * and loaders to ensure smooth ad operation while adhering to the required specifications.
 *
 * ## Responsibilities
 * - Parsing and managing bid responses from the ad server.
 * - Loading and preparing ad assets asynchronously.
 * - Handling user interactions such as ad impressions and clicks.
 * - Tracking analytics and reporting events to external systems.
 *
 * ## Components
 * - **AdLoader**: Responsible for loading the ad and parsing the bid response.
 * - **NativeAdAssetsProvider**: Manages the assets associated with the ad, such as images, videos, and text.
 * - **NativeAdTracker**: Tracks impressions and click events for reporting and analytics.
 * - **AdShowListener**: Tracks when the ad is successfully displayed to the user.
 *
 * ## Features
 * - Supports asynchronous ad loading with timeout handling.
 * - Provides callbacks for load success, failure, and timeout scenarios.
 * - Tracks impressions and user interactions for accurate analytics.
 * - Manages ad lifecycle, including initialization and destruction.
 */
internal class NativeAdImpl(
    private val adUnitId: String,
    private val nativeAdLoader: NativeAdLoader,
    override val assets: NativeAdAssetsProvider,
    private val appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val externalLinkHandler: ExternalLinkHandler,
    private val persistentHttpRequest: PersistentHttpRequest,
    private val createLoadTimeoutManager: AdCreateLoadTimeoutManager,
) : NativeAd, CreateAdObjectTime by createLoadTimeoutManager {
    override var interactionListener: NativeAd.InteractionListener? = null

    private val adFormatType = AdFormatType.NATIVE
    @VisibleForTesting
    val scope = CoroutineScope(DispatcherProvider().main)

    override val isLoaded: Boolean get() = assets.preparedAssets != null

    private val acmLoadTimerEvent: TimerEvent = acm.startTimerEvent(AcmTimer.LoadAd.eventName)

    @VisibleForTesting
    var adShowListenerWithTracker: NativeAdShowListenerWithTracker? = null

    @VisibleForTesting
    var loadJob: Job? = null

    /**
     * Loads a native ad using the provided bid response JSON.
     *
     * This method initiates the process of loading a native ad based on the given bid response.
     * Once an ad is successfully loaded ([isLoaded] becomes true), subsequent calls to this
     * method on the same `NativeAdImpl` instance will be ignored. To load a new ad, a new
     * `NativeAd` instance must be created using [Moloco.createNativeAd], and then its [load]
     * method should be invoked.
     */
    @Synchronized
    override fun load(bidResponseJson: String, listener: AdLoad.Listener?) {
        if (loadJob?.isActive == true) {
            MolocoLogger.warn(TAG, "load() called while another load operation is in progress. Ignoring this call.")
            return
        }

        if (isLoaded) {
            MolocoLogger.warn(TAG, "load() called but ad is already loaded. Ignoring this call.")
            return
        }

        loadJob = scope.launch {
            val adLoadListenerWithTracker = createAdLoadListenerWithTracker(acmLoadTimerEvent, listener)


            // Loads native ad and in the event of failures, will notify the `adLoadListenerWithTracker`
            val loadedNativeAd = nativeAdLoader.load(
                bidResponseJson,
                acmLoadTimerEvent,
                adLoadListenerWithTracker
            ).getOrElse {
                MolocoLogger.warn(TAG, "Failed to load native ad.", it)
                return@launch
            }

            // Assets are fetched, now we can create show listeners and populate the [assets]
            with(loadedNativeAd) {
                adShowListenerWithTracker = NativeAdShowListenerWithTracker(
                    adUnitId = adUnitId,
                    bid = bid,
                    ortbResponse = ortbResponse,
                    appLifecycleTrackerService = appLifecycleTrackerService,
                    customUserEventBuilderService = customUserEventBuilderService,
                    adFormatType = adFormatType,
                    persistentHttpRequest = persistentHttpRequest,
                    externalLinkHandler = externalLinkHandler,
                )
                assets.preparedAssets = preparedAssets
                assets.onClick = ::handleGeneralAdClick

                adLoadListenerWithTracker.onAdLoadSuccess(createAdInfo(adUnitId, bid.price), bid.ext.sdkEvents)
            }
        }
    }

    private fun createAdLoadListenerWithTracker(acmLoadTimerEvent: TimerEvent, listener: AdLoad.Listener?) = AdLoadListenerTracker(
        originListener = listener,
        acmLoadTimerEvent = acmLoadTimerEvent,
        adFormatType = adFormatType
    )

    override fun handleImpression() {
        interactionListener?.onImpressionHandled()

        adShowListenerWithTracker?.fireImpressionListenerAndTracker()
    }

    override fun handleGeneralAdClick() {
        interactionListener?.onGeneralClickHandled()

        adShowListenerWithTracker?.fireClickListenerAndTracker()
    }

    override fun destroy() {
        scope.cancel()
        assets.destroy()
        interactionListener = null
    }

    companion object {
        private const val TAG = "NativeAdImpl"
    }
}
