package com.moloco.sdk.internal.publisher.nativead

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.acm.TimerEvent
import com.moloco.sdk.internal.MolocoInternalAdError
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmResultTag
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.createInternalAdErrorInfo
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.ortb.model.SdkEvents
import com.moloco.sdk.internal.publisher.AdCreateLoadTimeoutManager
import com.moloco.sdk.internal.publisher.BidLoader
import com.moloco.sdk.internal.publisher.InternalAdLoadListener
import com.moloco.sdk.internal.publisher.nativead.model.NativeOrtbResponse
import com.moloco.sdk.internal.publisher.nativead.model.PreparedNativeAssets
import com.moloco.sdk.internal.publisher.nativead.parser.NativeAdOrtbResponseParser
import com.moloco.sdk.internal.publisher.nativead.parser.PrepareNativeAssetException
import com.moloco.sdk.internal.publisher.nativead.parser.prepareNativeAssets
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.TimeProviderService
import com.moloco.sdk.internal.services.WebViewAvailabilityChecker
import com.moloco.sdk.publisher.AdFormatType
import com.moloco.sdk.publisher.MolocoAdError
import com.moloco.sdk.publisher.MolocoAdError.ErrorType
import com.moloco.sdk.publisher.createAdInfo
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.AdLoadTimeoutError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.NativeAdLoadError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.errors.WebViewAdError
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeoutOrNull
import kotlin.time.Duration
import kotlin.time.DurationUnit
import kotlin.time.toDuration

/**
 * A stateless loader class responsible for handling the end-to-end process of loading and preparing native ads.
 *
 * This includes parsing bid responses, managing timeouts, and preparing assets required for rendering.
 * It interacts with various components such as `BidLoader`, `AdCreateLoadTimeoutManager`, and `AndroidClientMetrics`
 * to ensure that native ads are loaded efficiently and correctly.
 *
 * ## Responsibilities
 * - **Bid Parsing**: Parses the JSON bid response (`adm`) to extract valid bids.
 * - **Asset Preparation**: Prepares the necessary assets (e.g., images, videos) for rendering native ads.
 * - **Timeout Handling**: Monitors and enforces load timeouts for asset preparation and ad loading.
 * - **Metrics and Tracking**: Records metrics for ad load attempts, successes, and failures.
 *
 * ## Lifecycle
 * 1. **Initialization**: The class is initialized with dependencies such as the Android `Context`, ad unit ID,
 *    bid loader, and metrics trackers.
 * 2. **Ad Loading**: The `load` function is called with bid response data and listeners to handle the ad loading process.
 * 3. **Error Handling**: Errors during bid parsing or asset preparation are recorded, and appropriate callbacks are invoked.
 *
 * ## Key Dependencies
 * - **[BidLoader]**: Used to parse and validate bid responses.
 * - **[AdCreateLoadTimeoutManager]**: Determines appropriate timeouts for asset preparation and ad creation.
 * - **[AndroidClientMetrics]**: Records and tracks performance and outcomes for ad loading operations.
 *
 */
internal class NativeAdLoader(
    private val context: Context,
    private val adUnitId: String,
    private val bidLoader: BidLoader,
    private val ortbResponseParser: NativeAdOrtbResponseParser,
    private val createLoadTimeoutManager: AdCreateLoadTimeoutManager,
    private val acm: AndroidClientMetrics,
    private val timeProvider: TimeProviderService,
    private val webViewChecker: WebViewAvailabilityChecker,
) {
    private val adFormatType = AdFormatType.NATIVE
    private val createToLoadTimerEvent: TimerEvent = acm.startTimerEvent(
        AcmTimer.CreateToLoad.eventName).withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())

    private val dispatcherProvider = DispatcherProvider()

    /**
     * Initiates the process of loading a native ad using the provided bid response data.
     *
     * This function processes the bid response JSON to extract necessary information, fetches
     * and prepares ad assets, and notifies the listener about the progress and result of the ad load.
     * The listener is responsible for handling key events such as:
     * - Receiving a callback when the ad loading starts.
     * - Handling success scenarios by receiving the fully loaded ad details.
     * - Responding to failure scenarios by receiving error information.
     *
     * The loading process is tracked and measured for performance, and any errors encountered
     * during bid parsing, asset fetching, or other stages are reported through the listener.
     *
     * The function uses asynchronous operations and runs within the default dispatcher context.
     */
    suspend fun load(
        bidResponseJson: String,
        acmLoadTimerEvent: TimerEvent,
        adLoadListenerWithTracker: InternalAdLoadListener,
    ): kotlin.Result<LoadedNativeAd> = withContext(dispatcherProvider.default) {
        webViewChecker.checkAvailability().onFailure {
            MolocoLogger.error(TAG, "WebView Error: ${it.message}", it, true)
            adLoadListenerWithTracker.onAdLoadFailed(
                createInternalAdErrorInfo(adUnitId, MolocoAdError.ErrorType.AD_LOAD_WEBVIEW_FAILED, WebViewAdError.WEBVIEW_NOT_AVAILABLE_ERROR), null
            )
            return@withContext kotlin.Result.failure(it)
        }

        val adLoadStartTimeMs = timeProvider.currentTime()

        // Analytics
        acmLoadTimerEvent.startTimer()
        // Recording of this timer is as close to the public API as we can get.
        acm.recordTimerEvent(createToLoadTimerEvent)
        acm.recordCountEvent(
            CountEvent(AcmCount.LoadAdAttempt.eventName)
                .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
        )

        val bid: Bid = handleBidParsing(
            bidResponseJson,
            acmLoadTimerEvent,
            adLoadListenerWithTracker
        ).getOrElse {
            return@withContext kotlin.Result.failure(it)
        }

        withContext(dispatcherProvider.main) {
            adLoadListenerWithTracker.onAdLoadStarted(
                createAdInfo(adUnitId, bid.price),
                adLoadStartTimeMs,
                bid.ext.sdkEvents)
        }

        val ortbResponse: NativeOrtbResponse = handleOrtbParsing(
            bid.adm,
            bid.ext.sdkEvents,
            adLoadListenerWithTracker
        ).getOrElse { return@withContext kotlin.Result.failure(it) }

        val preparedAssets = handleAssetsFetching(
            bid.ext.sdkEvents,
            ortbResponse,
            adLoadListenerWithTracker,
            adLoadStartTimeMs
        ).getOrElse { return@withContext kotlin.Result.failure(it) }

        return@withContext kotlin.Result.success(LoadedNativeAd(bid, ortbResponse, preparedAssets))
    }

    private suspend fun handleBidParsing(bidResponseJson: String, acmLoadTimerEvent: TimerEvent, adLoadTracker: InternalAdLoadListener) =
        bidLoader.parse(adUnitId, bidResponseJson).let {
            when (it) {
                is Result.Failure -> {
                    acm.recordTimerEvent(
                        acmLoadTimerEvent
                            .withTag(AcmTag.Result.tagName, "failure")
                            .withTag(AcmTag.Reason.tagName, "${ErrorType.AD_BID_PARSE_ERROR.errorCode}")
                            .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
                    )

                    acm.recordCountEvent(
                        CountEvent(AcmCount.LoadAdFailed.eventName)
                            .withTag(AcmTag.Reason.tagName, "${ErrorType.AD_BID_PARSE_ERROR.errorCode}")
                            .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
                    )

                    fireAdLoadFailedEventOnUiThread(adLoadTracker, it.value, null)
                    kotlin.Result.failure(Exception(it.value.toString()))
                }

                is Result.Success -> kotlin.Result.success(it.value)
            }
        }

    /**
     * Parses the bid response (`adm`) and converts it into a [NativeOrtbResponse].
     */
    private suspend fun handleOrtbParsing(adm: String, sdkEvents: SdkEvents?, adLoadTracker: InternalAdLoadListener): kotlin.Result<NativeOrtbResponse> {
        val result = ortbResponseParser.parseNativeOrtbResponse(adm)
        result.onFailure {
            MolocoLogger.error(TAG, "handleOrtbParsing", it)
            val error = createInternalAdErrorInfo(adUnitId, ErrorType.AD_LOAD_FAILED, NativeAdLoadError.NATIVE_AD_ORTB_RESPONSE_NULL_ERROR)
            fireAdLoadFailedEventOnUiThread(adLoadTracker, error, sdkEvents)
        }
        return result
    }

    @VisibleForTesting
    suspend fun handleAssetsFetching(
        sdkEvents: SdkEvents?,
        ortbResponse: NativeOrtbResponse,
        adLoadTracker: InternalAdLoadListener,
        adLoadStartTimeMs: Long,
    ): kotlin.Result<PreparedNativeAssets> {
        MolocoLogger.info(TAG, "nativeAd load has $adLoadStartTimeMs to load the ad")
        val timeout = createLoadTimeoutManager.calculateTimeout(adLoadStartTimeMs)
        val result = withTimeoutOrNull(timeout) {
            fetchAssets(ortbResponse, timeout)
        }
        MolocoLogger.info(TAG, "Handling native ad load result: $result")

        return when (result) {
            null -> {
                MolocoLogger.warn(TAG, "Native ad load timeout")
                val timeoutError = createInternalAdErrorInfo(
                    adUnitId = adUnitId,
                    errorType = ErrorType.AD_LOAD_TIMEOUT_ERROR,
                    subErrorType = AdLoadTimeoutError.NATIVE_AD_LOAD_INTERNAL_TIMEOUT_ERROR
                )
                fireAdLoadFailedEventOnUiThread(adLoadTracker, timeoutError, sdkEvents)

                kotlin.Result.failure(Exception(timeoutError.toString()))
            }

            is Result.Failure -> {
                val error = createInternalAdErrorInfo(adUnitId, ErrorType.AD_LOAD_FAILED, result.value)
                fireAdLoadFailedEventOnUiThread(adLoadTracker, error, sdkEvents)
                kotlin.Result.failure(Exception(error.toString()))
            }

            is Result.Success -> {
                kotlin.Result.success(result.value)
            }
        }
    }

    private suspend fun fetchAssets(ortbResponse: NativeOrtbResponse, timeout: Duration): Result<PreparedNativeAssets, MolocoAdSubErrorType> {
        val startTime = timeProvider.currentTime()

        /**
         * The general timeout for preparing NativeAd assets is defined in [NativeAdImpl.load].
         * The timeout is adjusted to account for the elapsed time, ensuring that the top-level timeout
         * is not triggered prematurely during this process.
         */
        val elapsedTime = timeProvider.currentTime() - startTime
        val remainingTimeout = (timeout.inWholeMilliseconds - elapsedTime).coerceAtLeast(0).toDuration(DurationUnit.MILLISECONDS)

        // We log additional metrics for native ads as we need a sub ad format type (which our current load flow doesn't easily support)
        val adType = if (ortbResponse.assets.filterIsInstance<NativeOrtbResponse.Asset.Video>().isNotEmpty()) "video" else "image"
        val nativeAdPrepareTime = acm.startTimerEvent(AcmTimer.NativePrepareAd.eventName)

        acm.recordCountEvent(
            CountEvent(AcmCount.NativeAdLoadAdAttempted.eventName)
                .withTag(AcmTag.AdType.tagName, adType.lowercase())
        )

        return when (val preparedAssetsResult = prepareAssets(ortbResponse.assets, remainingTimeout)) {
            is Result.Failure -> {
                MolocoLogger.info(TAG, "NativeAd load failed: ${preparedAssetsResult.value.errorSubType}")
                acm.recordCountEvent(
                    CountEvent(AcmCount.NativeLoadAd.eventName)
                        .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                        .withTag(AcmTag.AdType.tagName, adType.lowercase())
                        .withTag(AcmTag.Reason.tagName, preparedAssetsResult.value.errorSubType.metricsRepresentation)
                        .withTag("asset_id", preparedAssetsResult.value.assetId.toString())
                )
                acm.recordTimerEvent(
                    nativeAdPrepareTime
                        .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                        .withTag(AcmTag.AdType.tagName, adType.lowercase())
                        .withTag(AcmTag.Reason.tagName, preparedAssetsResult.value.errorSubType.metricsRepresentation)
                        .withTag("asset_id", preparedAssetsResult.value.assetId.toString())
                )
                Result.Failure(preparedAssetsResult.value.errorSubType)
            }

            is Result.Success -> {
                MolocoLogger.info(TAG, "NativeAd load successfully parsed and loaded all assets")
                acm.recordCountEvent(
                    CountEvent(AcmCount.NativeLoadAd.eventName)
                        .withTag(AcmTag.Result.tagName, AcmResultTag.success.name)
                        .withTag(AcmTag.AdType.tagName, adType.lowercase())
                )
                acm.recordTimerEvent(
                    nativeAdPrepareTime
                        .withTag(AcmTag.Result.tagName, AcmResultTag.success.name)
                        .withTag(AcmTag.AdType.tagName, adType.lowercase())
                )
                Result.Success(preparedAssetsResult.value)
            }
        }
    }

    /**
     * Prepares native ad assets from the given list of [NativeOrtbResponse.Asset].
     *
     * @param assets A list of native ad assets provided in the ORTB response.
     * @return A [PreparedNativeAssets] object containing the prepared assets, or `null` if preparation fails.
     */
    private suspend fun prepareAssets(
        assets: List<NativeOrtbResponse.Asset>,
        timeout: Duration,
    ): Result<PreparedNativeAssets, PrepareNativeAssetException> =
        when (val result = prepareNativeAssets(context, assets, timeout)) {
            is Result.Success -> Result.Success(result.value)
            is Result.Failure -> {
                MolocoLogger.error(TAG, "NativeAd prepareAssets failed", result.value)
                Result.Failure(result.value)
            }
        }

    private suspend fun fireAdLoadFailedEventOnUiThread(
        adLoadTracker: InternalAdLoadListener,
        error: MolocoInternalAdError,
        sdkEvents: SdkEvents?,
    ) = withContext(dispatcherProvider.main) {
        adLoadTracker.onAdLoadFailed(error, sdkEvents)
    }

    data class LoadedNativeAd(val bid: Bid, val ortbResponse: NativeOrtbResponse, val preparedAssets: PreparedNativeAssets)
    companion object {
        private const val TAG = "NativeAdLoader"
    }
}
