package com.moloco.sdk.internal.services

import android.content.Context
import android.provider.Settings
import android.view.accessibility.AccessibilityManager
import android.view.accessibility.CaptioningManager
import com.moloco.sdk.internal.MolocoLogger

internal interface AccessibilityInfoService {
    /**
     * Get the font scale of the device
     */
    fun getFontScale(): Float

    /**
     * Check if the large pointer icon is enabled
     */
    fun isLargePointerIconEnabled(): Boolean

    /**
     * Check if the reduce bright colors is activated
     */
    fun isReduceBrightColorsActivated(): Boolean

    /**
     * Check if the accessibility captioning is enabled
     */
    fun isAccessibilityCaptioningEnabled(): Boolean
}

internal class AccessibilityInfoServiceImpl(private val context: Context) : AccessibilityInfoService {
    override fun getFontScale(): Float {
        val accessibilityManager = context.getSystemService(Context.ACCESSIBILITY_SERVICE) as AccessibilityManager
        return if (accessibilityManager.isEnabled) {
            context.resources.configuration.fontScale
        } else {
            1.0f // Default scaling factor
        }
    }

    override fun isLargePointerIconEnabled(): Boolean {
        val accessibilityManager = context.getSystemService(Context.ACCESSIBILITY_SERVICE) as AccessibilityManager
        return if (accessibilityManager.isEnabled) {
            Settings.Secure.getInt(
                context.contentResolver,
                "accessibility_large_pointer_icon",
                0
            ) == 1
        } else {
            false
        }
    }

    override fun isReduceBrightColorsActivated(): Boolean {
        return try {
            Settings.Secure.getInt(
                context.contentResolver,
                "reduce_bright_colors_activated"
            ) == 1
        } catch (e: Settings.SettingNotFoundException ) {
            false
        } catch (e: SecurityException){
            // Android S+ doesn't allow access to this setting, hence the extra catch clause
            false
        }
    }

    override fun isAccessibilityCaptioningEnabled(): Boolean {
        return try {
            val captioningManager = context.getSystemService(Context.CAPTIONING_SERVICE) as CaptioningManager
            captioningManager.isEnabled
        } catch (e: Exception) {
            // Android L crashes on certain devices - https://mlc.atlassian.net/browse/SDK-3625
            false
        }
    }

    companion object {
        private const val TAG = "AccessibilityInfoService"
    }
}
