package com.moloco.sdk.internal.services.config.handlers

import com.moloco.sdk.Init
import com.moloco.sdk.internal.configs.DefaultOperationalMetricsConfig
import com.moloco.sdk.internal.configs.OperationalMetricsConfig

/**
 * Config Handler for Operational Metrics - Android Client Metrics Library.
 *
 * This handler is responsible for parsing and providing the operational metrics
 * configuration from the SDK initialization response. It creates an instance of
 * [OperationalMetricsConfig] based on the provided response or uses the default
 * [DefaultOperationalMetricsConfig] if the operational metrics configuration is not present.
 */
internal class OperationalMetricsConfigHandler: ConfigHandler {

    /**
     * Handles the operational metrics configuration from the SDK initialization response.
     *
     * Parses the [sdkInitResponse] to extract the operational metrics configuration.
     * If the configuration is present in the response, it creates an [OperationalMetricsConfig]
     * object. Otherwise, it returns the [DefaultOperationalMetricsConfig].
     *
     * @param sdkInitResponse The SDK initialization response containing the operational metrics configuration.
     * @return The parsed operational metrics configuration object or the default configuration.
     */
    override fun handleConfig(sdkInitResponse: Init.SDKInitResponse): OperationalMetricsConfig {
        val operationalMetricsConfig = if (sdkInitResponse.hasOperationalMetricsConfig()) {
            val operationalMetricsConfig = sdkInitResponse.operationalMetricsConfig
            OperationalMetricsConfig(
                reportingUrl = operationalMetricsConfig.url,
                pollingIntervalSeconds = operationalMetricsConfig.pollingIntervalSeconds
            )
        } else {
            DefaultOperationalMetricsConfig
        }

        return operationalMetricsConfig
    }


    /**
     * Returns the class type of the operational metrics configuration object.
     *
     * @return The class type of [OperationalMetricsConfig].
     */
    override fun getConfigType() = OperationalMetricsConfig::class.java
}
