package com.moloco.sdk.internal.unity_bridge.internal

import android.os.Handler
import android.os.Looper
import com.moloco.sdk.internal.unity_bridge.MolocoUnityLoadCallback
import com.moloco.sdk.internal.unity_bridge.MolocoUnityShowCallback
import com.moloco.sdk.publisher.AdLoad
import com.moloco.sdk.publisher.InterstitialAd
import com.moloco.sdk.publisher.InterstitialAdShowListener
import com.moloco.sdk.publisher.Moloco
import com.moloco.sdk.publisher.MolocoAd
import com.moloco.sdk.publisher.MolocoAdError
import io.ktor.util.collections.ConcurrentMap

/**
 * Manages ads for Unity integration with the Moloco SDK.
 *
 * This class serves as a bridge between Unity and the native Moloco SDK, handling the
 * creation, loading, and presentation of ads. It maintains a thread-safe map
 * of loaded ads and ensures all callbacks are properly delivered on the appropriate thread.
 *
 * All public methods in this class safely handle threading with message posting to ensure
 * Unity callbacks execute on the correct thread.
 */
internal class UnityAdManager {
    /**
     * Thread-safe map storing loaded ads by their ad unit IDs.
     */
    private val loadedInterstitialMap = ConcurrentMap<String, InterstitialAd>()

    /**
     * Creates and loads an ad for the specified ad unit ID.
     *
     * This method first creates an ad instance, then loads it with the provided bid response.
     * All callbacks are posted to the original calling thread. If an ad for the specified ad unit ID
     * already exists and is loaded, it will be reused.
     *
     * @param adUnitId The unique identifier for the ad unit
     * @param bidResponse The bid response JSON string containing ad content details
     * @param unityLoadCallback Callback interface to notify Unity of ad load events
     * @throws IllegalStateException If called from a thread that hasn't called Looper.prepare()
     */
    fun loadInterstitial(
        adUnitId: String,
        bidResponse: String,
        unityLoadCallback: MolocoUnityLoadCallback,
    ) {
        val handler = createHandler()

        Moloco.createInterstitial(adUnitId) { ad, error ->
            if (error != null) {
                handler.post {
                    unityLoadCallback.onAdLoadFailed(adUnitId, error.toString())
                }
                return@createInterstitial
            }

            loadInterstitialImpl(ad as InterstitialAd, adUnitId, bidResponse, handler, unityLoadCallback)
        }
    }

    /**
     * Shows a previously loaded ad.
     *
     * Attempts to display the ad identified by the provided ad unit ID. If the ad isn't loaded
     * or doesn't exist, an error is reported via the callback. All event callbacks are posted
     * to the original calling thread.
     *
     * @param adUnitId The unique identifier for the ad unit to display
     * @param callback Callback interface to notify Unity of ad display events
     * @throws IllegalStateException If called from a thread that hasn't called Looper.prepare()
     */
    fun showInterstitial(adUnitId: String, callback: MolocoUnityShowCallback) {
        val handler = createHandler()

        val ad = loadedInterstitialMap[adUnitId]
        if (ad == null || !ad.isLoaded) {
            ad?.destroy()
            handler.post {
                callback.onAdShowFailed(adUnitId, "Ad cannot be shown as it was not loaded")
            }
            return
        }

        ad.show(object : InterstitialAdShowListener {
            override fun onAdShowSuccess(molocoAd: MolocoAd) {
                handler.post {
                    callback.onAdShowSuccess(molocoAd.adUnitId)
                }
            }

            override fun onAdShowFailed(molocoAdError: MolocoAdError) {
                handler.post {
                    callback.onAdShowFailed(molocoAdError.adUnitId, molocoAdError.toString())
                }
            }

            override fun onAdHidden(molocoAd: MolocoAd) {
                handler.post {
                    callback.onAdHidden(molocoAd.adUnitId)
                }
            }

            override fun onAdClicked(molocoAd: MolocoAd) {
                handler.post {
                    callback.onAdClicked(molocoAd.adUnitId)
                }
            }
        })
    }

    /**
     * Implementation for loading an ad instance.
     *
     * This method handles the actual loading process after the ad has been created. It manages
     * the state of existing ads, cleans up unneeded instances, and updates the ad storage map.
     *
     * @param ad The ad instance to load
     * @param adUnitId The unique identifier for the ad unit
     * @param bidResponse The bid response JSON string containing ad content details
     * @param handler Handler for posting callbacks to the original thread
     * @param unityLoadCallback Callback interface to notify Unity of ad load events
     */
    private fun loadInterstitialImpl(
        ad: InterstitialAd,
        adUnitId: String,
        bidResponse: String,
        handler: Handler,
        unityLoadCallback: MolocoUnityLoadCallback,
    ) {
        loadedInterstitialMap[adUnitId]?.let {
            if (it.isLoaded) {
                handler.post {
                    unityLoadCallback.onAdLoadSuccess(adUnitId)
                }
                return
            } else {
                it.destroy()
            }
        }

        ad.load(bidResponse, object : AdLoad.Listener {
            override fun onAdLoadSuccess(molocoAd: MolocoAd) {
                loadedInterstitialMap[adUnitId] = ad
                handler.post {
                    unityLoadCallback.onAdLoadSuccess(adUnitId)
                }
            }

            override fun onAdLoadFailed(molocoAdError: MolocoAdError) {
                handler.post {
                    unityLoadCallback.onAdLoadFailed(adUnitId, molocoAdError.toString())
                }
            }
        })
    }

    /**
     * Creates a Handler associated with the current thread's Looper.
     *
     * Used to ensure callbacks are delivered on the same thread that called the initial method.
     *
     * @return A Handler for the current thread
     * @throws IllegalStateException If called from a thread without a Looper
     */
    private fun createHandler() = run {
        val looper = Looper.myLooper()
        if (looper == null) {
            throw IllegalStateException(("Can't create handler inside thread " + Thread.currentThread()
                + " that has not called Looper.prepare()"))
        } else {
            Handler(looper)
        }
    }
}
