package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer

import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Shader
import android.graphics.drawable.BitmapDrawable
import android.os.Build
import android.util.Base64
import android.view.View
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.semantics.contentDescription
import androidx.compose.ui.semantics.semantics
import androidx.compose.ui.semantics.testTag
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.service_locator.SdkObjectFactory.context

/**
 * Interface for applying a watermark.
 */
internal interface Watermark {
    /**
     * Applies a watermark from a base64 encoded string to a Composable Unit.
     */
    @Composable
    fun ApplyWatermark(content: @Composable () -> Unit)

    /**
     * Applies a watermark from a base64 encoded string to a View.
     */
    fun applyWatermark(view: View)
}

internal class WatermarkImpl(
    private var base64String: String? = null
): Watermark {

    @Composable
    override fun ApplyWatermark(content: @Composable () -> Unit) {
        WatermarkComposable(base64String) {
            content()
        }
    }

    /**
     * Applies a watermark from a base64 encoded string to a View. Only on Android API 23+
     */
    override fun applyWatermark(view: View) {
        // If we don't have the watermark string, just don't apply it
        val overlayBitMap: Bitmap = createBitmapFromBase64(base64String) ?: return

        val overlayRepeat = BitmapDrawable(context.resources, overlayBitMap)
        overlayRepeat.setTileModeXY(Shader.TileMode.REPEAT, Shader.TileMode.REPEAT)

        val displayMetrics = context.resources.displayMetrics
        overlayRepeat.setTargetDensity(displayMetrics)

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            view.foreground = overlayRepeat
        } else {
            view.background = overlayRepeat
        }

        view.isClickable = false
        view.isFocusable = false
    }
}

@Composable
internal fun WatermarkComposable(
    base64String: String?,
    content: @Composable () -> Unit
) {

    val contentDescription = "Watermark Overlay"
    val overlayImageBitmap = remember(base64String) {
        createImageBitmapFromBase64(base64String)
    }

    // If no watermark, simply show the content
    if (overlayImageBitmap == null) {
        content()
        return
    }

    Box (
        modifier = Modifier
            .fillMaxSize()
            .semantics {
                contentDescription.let {
                    this.contentDescription = it
                    this.testTag = it
                }
            }
    ) {
        content()
        Canvas(modifier = Modifier.fillMaxSize()) {
            val imageWidth = overlayImageBitmap.width.toFloat()
            val imageHeight = overlayImageBitmap.height.toFloat()

            val canvasWidth = size.width
            val canvasHeight = size.height

            var x = 0f
            while (x < canvasWidth) {
                var y = 0f
                while (y < canvasHeight) {
                    drawImage(
                        image = overlayImageBitmap,
                        topLeft = Offset(x, y)
                    )
                    y += imageHeight
                }
                x += imageWidth
            }
        }
    }
}

/**
 * Creates a [Bitmap] from a Base64 encoded string.
 *
 * This function decodes the provided Base64 encoded string into a byte array and then
 * attempts to decode that byte array into a [Bitmap]. If the Base64 string is null or
 * empty, or if the decoding fails, an error is logged and the function returns `null`.
 *
 * @param base64String The Base64 encoded string representing the bitmap data. It can be `null` or empty.
 * @return The decoded [Bitmap] if successful, or `null` if there was an error or the input was invalid.
 */
internal fun createBitmapFromBase64(base64String: String?): Bitmap? {
    return try {
        if (base64String.isNullOrEmpty()) {
            MolocoLogger.debug("BitmapCreationError", "Base64 string is null or empty")
            return null
        }

        val overlayBytes = Base64.decode(base64String, Base64.DEFAULT)
        val bitmap = BitmapFactory.decodeByteArray(overlayBytes, 0, overlayBytes.size)

        if (bitmap == null) {
            MolocoLogger.error("BitmapCreationError", "BitmapFactory failed to decode the byte array")
        }

        bitmap
    } catch (e: Exception) {
        MolocoLogger.error("BitmapCreationError", "Error creating bitmap from base64", e)
        null
    }
}

/**
 * Creates an [ImageBitmap] from a Base64 encoded string.
 *
 * This function first calls [createBitmapFromBase64] to decode the Base64 encoded string into a [Bitmap],
 * and then converts that [Bitmap] into an [ImageBitmap]. If the Base64 string is invalid or an error occurs
 * during the decoding process, the function returns `null`.
 *
 * @param base64String The Base64 encoded string representing the bitmap data. It can be `null` or empty.
 * @return The decoded [ImageBitmap] if successful, or `null` if there was an error or the input was invalid.
 */
internal fun createImageBitmapFromBase64(base64String: String?): ImageBitmap? {
    return createBitmapFromBase64(base64String)?.asImageBitmap()
}
