package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import android.content.Context
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdWebViewOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.FullscreenAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MraidAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.toMraidFullscreenError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidActivity
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidBaseAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidFullscreenContentController
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlin.time.Duration

internal fun MraidFullscreenAd(
    context: Context,
    watermark: Watermark,
    mraidAdLoader: MraidAdLoad,
    mraidBaseAd: MraidBaseAd,
    mraidFullscreenController: MraidFullscreenContentController,
    mraidAdActivity: MraidActivity.Companion
): FullscreenAd<AdShowListener, AdWebViewOptions> = MraidFullscreenAdImpl(
    context,
    watermark,
    mraidAdLoader,
    mraidBaseAd,
    mraidFullscreenController,
    mraidAdActivity
)

private class MraidFullscreenAdImpl(
    private val context: Context,
    private val watermark: Watermark,
    private val mraidAdLoader: MraidAdLoad,
    private val mraidBaseAd: MraidBaseAd,
    private val mraidFullscreenController: MraidFullscreenContentController,
    private val mraidActivity: MraidActivity.Companion
) : FullscreenAd<AdShowListener, AdWebViewOptions> {

    override val creativeType = CreativeType.MRAID

    private val scope = CoroutineScope(DispatcherProvider().main)

    private var adLoadListener: AdLoad.Listener? = null
    private var adShowListener: AdShowListener? = null
    private var showCalled = false

    override val isLoaded by mraidAdLoader::isLoaded

    override fun load(timeout: Duration, listener: AdLoad.Listener?) {
        adLoadListener = listener
        mraidBaseAd.onError = this::onError
        mraidAdLoader.load(timeout, listener)
    }

    private val _isAdDisplaying = MutableStateFlow(false)
    override val isAdDisplaying: StateFlow<Boolean> = _isAdDisplaying

    private val _isAdForciblyClosed = MutableStateFlow(false)
    override val isAdForciblyClosed: StateFlow<Boolean> = _isAdForciblyClosed

    override fun show(options: AdWebViewOptions, listener: AdShowListener?) {
        mraidBaseAd.onClick = { listener?.onClick() }
        adShowListener = listener

        showCalled = true
        val adData = when(val mraidAdResult = mraidAdLoader.mraidLoadResult) {
            is Result.Failure -> {
                val error = mraidAdResult.value
                listener?.onShowError(error)
                return
            }
            is Result.Success -> {
                mraidAdResult.value
            }
        }

        if (mraidActivity.show(adData, mraidFullscreenController, context, options, watermark, ::destroy, ::onForciblyClosed)) {
            _isAdDisplaying.value = true
        } else {
            listener?.onShowError(MraidAdError.MRAID_FULLSCREEN_AD_ACTIVITY_SHOW_FAILED_ERROR)
        }
    }

    private fun onError(error: MraidAdError) {
        if (showCalled) {
            adShowListener?.onShowError(error.toMraidFullscreenError())
        } else {
            adLoadListener?.onLoadError(error.toMraidFullscreenError())
        }
    }

    private fun onForciblyClosed() {
        _isAdForciblyClosed.value = true
    }

    override fun destroy() {
        scope.cancel()
        mraidBaseAd.destroy()
        // TODO. Refactor. Take Vast/Static implementation as example: suspendable show method wrapped with try finally block
        _isAdDisplaying.value = false
    }
}
