package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import android.content.Context
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.ClickthroughService
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.TemplateAdOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.TemplateAdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.TemplateFullscreenAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.ad.WebviewAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.ACMHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.ClickthroughEventHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.CloseHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.HttpEventHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.LaunchUrlHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.PlayListItemDisplayingEventHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.RequiredContentEventHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.fullscreen.FullscreenWebviewActivity
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlin.time.Duration

internal class TemplateFullscreenAdImpl(
    private val context: Context,
    adm: String,
    clickthroughService: ClickthroughService,
    private val watermark: Watermark,
    private val externalLinkHandler: ExternalLinkHandler,
    private val persistentHttpRequest: PersistentHttpRequest,
) : TemplateFullscreenAd {
    private val scope = CoroutineScope(DispatcherProvider().main)
    private val closeHandler: CloseHandler = CloseHandler(scope)
    private val contentLoadedHandler = RequiredContentEventHandler(scope)
    private val clickthroughEventHandler = ClickthroughEventHandler(clickthroughService, scope)
    private val playListItemDisplayingEventHandler = PlayListItemDisplayingEventHandler(scope)
    private val eventHandlers = setOf(
        closeHandler,
        clickthroughEventHandler,
        contentLoadedHandler,
        ACMHandler(),
        LaunchUrlHandler(externalLinkHandler),
        HttpEventHandler(persistentHttpRequest),
        playListItemDisplayingEventHandler
    )
    private val webviewAd = WebviewAd(context, adm, eventHandlers, clickthroughEventHandler, contentLoadedHandler, playListItemDisplayingEventHandler)

    override fun show(options: TemplateAdOptions, listener: TemplateAdShowListener?) {
        val delegateListener = object : TemplateAdShowListener {
            override fun onShowError(internalShowError: MolocoAdSubErrorType) {
                listener?.onShowError(internalShowError)
                destroy()
            }

            override fun onClick() {
                listener?.onClick()
            }
        }

        watermark.applyWatermark(webviewAd.webView)
        webviewAd.show(delegateListener)
        FullscreenWebviewActivity.show(
            context = context,
            isAdDisplayingEvent = _isAdDisplaying,
            webviewAd = webviewAd,
            isAdForciblyClosed = _isAdForciblyClosed
        )

        scope.launch {
            val eventHandlers = listOf(
                launch {
                    clickthroughEventHandler.clickthroughEvent.collect {
                        delegateListener.onClick()
                    }
                }
            )

            closeHandler.closeEvent.collectLatest { buttonClicked ->
                eventHandlers.onEach { it.cancel() }
                MolocoLogger.info(TAG, "Calling close()")
                FullscreenWebviewActivity.close()
            }
        }
    }

    override val creativeType: CreativeType? = null
    override val isLoaded: StateFlow<Boolean>
        get() = webviewAd.isLoaded

    override fun load(timeout: Duration, listener: AdLoad.Listener?) {
        webviewAd.load(timeout, listener)
    }

    private val _isAdDisplaying = MutableStateFlow(false)
    override val isAdDisplaying: StateFlow<Boolean> by lazy {
        // Added WebView's "isPageFinished" event check for static ads to prevent impression discrepancies,
        // between notifnt and html embedded trackers.
        _isAdDisplaying.combine(webviewAd.isAdDisplaying) { isAdDisplaying, webViewIsDisplaying ->
            MolocoLogger.info(TAG, "isAdDisplaying final: ${isAdDisplaying && webViewIsDisplaying}, _isAdDisplaying: $isAdDisplaying, webViewIsDisplaying: $webViewIsDisplaying")
            isAdDisplaying && webViewIsDisplaying
        }.stateIn(scope, SharingStarted.Eagerly, false)
    }
    private val _isAdForciblyClosed = MutableStateFlow<Boolean>(false)
    override val isAdForciblyClosed: StateFlow<Boolean>
        get() = _isAdForciblyClosed

    override fun destroy() {
        webviewAd.destroy()
    }

    companion object {
        private const val TAG = "TemplateFullscreenAd"
    }
}
