package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid

import android.content.Context
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Destroyable
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MraidAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.toMraidBannerError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.toMraidFullscreenError
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.async
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch

// TODO. Rename. Maybe. Refactor. Still ugly.
// Common logic for fs and banner mraid ads.
internal open class MraidBaseAd(
    context: Context,
    private val adm: String,
    private val mraidPlacementType: MraidPlacementType,
    var onClick: () -> Unit = {},
    var onError: (MraidAdError) -> Unit = {},
    private val externalLinkHandler: ExternalLinkHandler,
    // TODO. Refactor. Workaround for mraid endcard not crashing Vast player.
    private val forceBlockExpand: Boolean = false,
    val mraidBridge: MraidBridge,
) : Destroyable {

    private val TAG = "MraidBaseAd"
    private val scope = CoroutineScope(DispatcherProvider().main)
    private lateinit var mraidAdData: MraidAdData

    private var mraidViewState: MraidViewState? = null
        set(mraidViewState) {
            field = mraidViewState
            mraidViewState?.let { mraidBridge.sendMraidViewState(it) }
        }

    private val mraidViewVisualMetricsTracker =
        MraidViewVisualMetricsTracker(mraidBridge.webView, context, scope)

    private fun startListeningToVisualMetricsChanges() {
        mraidViewVisualMetricsTracker.isVisible.onEach {
            mraidBridge.sendMraidIsViewable(it)
        }.launchIn(scope)

        mraidViewVisualMetricsTracker.screenMetricsEvent.onEach {
            mraidBridge.sendMraidScreenMetrics(it.value)
        }.launchIn(scope)
    }

    private fun startListeningToErrors() {
        scope.launch {
            val error = mraidBridge.unrecoverableError.first { it != null }
            error?.let {
                when(mraidPlacementType) {
                    MraidPlacementType.Interstitial -> onError(it.toMraidFullscreenError())
                    MraidPlacementType.Inline -> onError(it.toMraidBannerError())
                }
            }
        }
    }

    private fun startListeningToMraidJsCommands() {
        mraidBridge.mraidJsCommands.onEach {
            when (it) {
                MraidJsCommand.Close -> handleMraidJsCommandClose()
                is MraidJsCommand.Open -> handleMraidJsCommandOpen(it)
                is MraidJsCommand.SetOrientationProperties -> {
                    // TODO. Refactor. Handled in the fullscreenController, skipped here.
                }
                is MraidJsCommand.Expand -> handleMraidJsCommandExpand(it)
                else -> mraidBridge.sendMraidError(
                    it,
                    "unsupported command: ${it.commandString}"
                )
            }
        }.launchIn(scope)
    }

    private fun handleMraidJsCommandClose() {
        if (mraidViewVisualMetricsTracker.isVisible.value) {
            closeFullscreenAdRepresentation()
        } else {
            mraidBridge.sendMraidError(
                MraidJsCommand.Close,
                "Can't close ad when mraid container is not visible to the user"
            )
        }
    }

    private fun handleMraidJsCommandOpen(openCmd: MraidJsCommand.Open) {
        if (mraidViewVisualMetricsTracker.isVisible.value) {
            externalLinkHandler(openCmd.uri.toString())
            onClick()
        } else {
            mraidBridge.sendMraidError(
                openCmd,
                "Can't open links when mraid container is not visible to the user"
            )
        }
    }

    private fun handleMraidJsCommandExpand(expandCmd: MraidJsCommand.Expand) {
        if (forceBlockExpand) {
            mraidBridge.sendMraidError(
                expandCmd,
                "expand() is force blocked for the current ad"
            )
            return
        }

        if (!mraidViewVisualMetricsTracker.isVisible.value) {
            mraidBridge.sendMraidError(
                expandCmd,
                "Can't expand() when mraid container is not visible to the user"
            )
            return
        }

        if (mraidViewState != MraidViewState.Default) {
            mraidBridge.sendMraidError(
                expandCmd,
                "In order to expand() mraid ad, container must be in Default view state"
            )
            return
        }

        if (mraidPlacementType == MraidPlacementType.Interstitial) {
            mraidBridge.sendMraidError(
                expandCmd,
                "expand() is not supported for interstitials"
            )
            return
        }

        // TODO. Support two-part expand.
        if (expandCmd.uri != null) {
            mraidBridge.sendMraidError(
                expandCmd,
                "Two-part expand is not supported yet"
            )
            return
        }

        onPreExpandAd()

        // TODO: Determine use of mraid expansion function. Current functionality is blocked
        // When expansion is called a call to show the MRAID activity was made. However, currently this path is never reached
        // 1. due to forceBlockExpand this call is blocked by the other MRAID Types
        // 2. additionally, this will not get triggered for interstitial types

        mraidViewState = MraidViewState.Expanded
    }

    open fun closeFullscreenAdRepresentation() {
        if (mraidViewState == MraidViewState.Expanded) {
            mraidViewState = MraidViewState.Default
        }
    }

    // Ugly workaround for inline mraid implementation to make sure
    // view is detached first before it's added to the fullscreen MraidAd.
    protected open fun onPreExpandAd() {}

    suspend fun loadAndReadyMraid(): Result<MraidAdData, MraidAdError> = scope.async {
        val loadMraidResult = mraidBridge.loadMraidHtml(adm)
        if (loadMraidResult is Result.Failure) {
            return@async loadMraidResult
        }

        with(mraidBridge) {
            sendMraidSupports(
                sms = false,
                tel = false,
                calendar = false,
                storePicture = false,
                inlineVideo = true
            )
            sendMraidPlacementType(mraidPlacementType)
            sendMraidIsViewable(mraidViewVisualMetricsTracker.isVisible.value)
            sendMraidScreenMetrics(mraidViewVisualMetricsTracker.screenMetricsEvent.value.value)

            mraidViewState = MraidViewState.Default

            startListeningToErrors()
            startListeningToMraidJsCommands()
            startListeningToVisualMetricsChanges()

            sendMraidReady()
        }

        mraidAdData = when(loadMraidResult) {
            is Result.Success -> {
                MolocoLogger.info(TAG, "Mraid Html data successfully loaded")
                loadMraidResult.value
            }

            is Result.Failure -> {
                MolocoLogger.error(TAG, "Mraid Html data load failed.")
                MraidAdData()
            }
        }
        loadMraidResult
    }.await()

    override fun destroy() {
        scope.cancel()
        mraidBridge.destroy()
        mraidViewVisualMetricsTracker.destroy()
    }
}
