package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid

import android.net.Uri
import android.text.TextUtils
import com.moloco.sdk.internal.Result

internal sealed class MraidJsCommand(val commandString: String) {
    object Close : MraidJsCommand(CLOSE)

    class Expand(val uri: Uri?) : MraidJsCommand(EXPAND)
    class Open(val uri: Uri) : MraidJsCommand(OPEN)

    class Resize(
        val widthDp: Int,
        val heightDp: Int,
        val offsetX: Int,
        val offsetY: Int,
        val allowOffscreen: Boolean,
    ) : MraidJsCommand(RESIZE)

    class SetOrientationProperties(
        val allowOrientationChange: Boolean,
        val forceOrientation: MraidOrientation,
    ) : MraidJsCommand(SET_ORIENTATION_PROPERTIES)

    fun SetOrientationProperties.toTemplateOrientationSettings() =
        com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.ad.orientation.OrientationSettings(
            forceOrientation.toAdOrientation(),
            allowOrientationChange,
        )

    companion object {

        private const val CLOSE = "close"
        private const val EXPAND = "expand"
        private const val OPEN = "open"
        private const val RESIZE = "resize"
        private const val SET_ORIENTATION_PROPERTIES = "setOrientationProperties"

        // TODO. Refactor?
        internal class Error(
            val isMraidScheme: Boolean,
            val description: String,
        )

        // TODO. Move from enum companion object?
        fun from(url: String?): Result<MraidJsCommand, Error> {
            val uri =
                kotlin.runCatching { Uri.parse(url) }.getOrNull()
                    ?: return Result.Failure(Error(false, "Invalid url: $url"))

            if (uri.scheme != "mraid") {
                return Result.Failure(Error(false, "Non-mraid url scheme: $url"))
            }

            val queryParams = uri.queryParams

            val res = when (uri.host) {
                CLOSE -> Close
                EXPAND -> tryCreateExpand(queryParams)
                OPEN -> tryCreateOpen(queryParams)
                RESIZE -> tryCreateResize()
                SET_ORIENTATION_PROPERTIES -> tryCreateSetOrientationProperties(queryParams)
                else -> null
            }

            return res?.let { Result.Success(it) }
                ?: Result.Failure(Error(true, "Unknown/unsupported mraid command ${uri.host}"))
        }

        private fun tryCreateExpand(queryParams: Map<String, String>): Expand? {
            val rawExpandUrl = queryParams["url"]

            return Expand(
                if (rawExpandUrl == null) {
                    null
                } else {
                    kotlin.runCatching {
                        Uri.parse(rawExpandUrl)
                    }.getOrNull()
                }
            )
        }

        private fun tryCreateOpen(queryParams: Map<String, String>): Open? {
            val rawOpenUrl = queryParams["url"] ?: return null

            return kotlin.runCatching {
                Open(Uri.parse(rawOpenUrl))
            }.getOrNull()
        }

        private fun tryCreateResize(): Resize? {
            // TODO. Implement when resize banner placements are allowed.
            return null
        }

        private fun tryCreateSetOrientationProperties(
            queryParams: Map<String, String>
        ): SetOrientationProperties? {
            val allowOrientationChange =
                queryParams["allowOrientationChange"]?.toBooleanStrictOrNull()
                    ?: return null

            val forceOrientation =
                MraidOrientation.from(queryParams["forceOrientation"])
                    ?: return null

            return SetOrientationProperties(allowOrientationChange, forceOrientation)
        }

        private val Uri.queryParams: Map<String, String>
            get() {
                val params = mutableMapOf<String, String>()
                for (queryParam in queryParameterNames) {
                    params[queryParam] = TextUtils.join(",", getQueryParameters(queryParam))
                }
                return params
            }
    }
}
