package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Color
import android.view.MotionEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.ClickthroughService
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.BaseWebView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.StaticAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.loadDataWithDefaultBaseUrl
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer.model.AdTouch
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.ButtonRecorder
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.ButtonTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.webview.HTMLCSSFixer
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeoutOrNull
import kotlin.time.Duration

private const val TAG = "StaticWebView"

// TODO. Ugly. Duplication of VastWebView.
//  Review settings during webview build/init.
@SuppressLint("SetJavaScriptEnabled", "ViewConstructor")
internal class StaticWebView(
    context: Context,
    private val scope: CoroutineScope = CoroutineScope(DispatcherProvider().main),
    private val clickthroughService: ClickthroughService,
    private val buttonTracker: ButtonTracker,
    private val isBannerClickthroughExperiment: Boolean = false,
    private val webViewClientImpl: WebViewClientImpl = WebViewClientImpl(
        scope,
        clickthroughService,
        buttonTracker
    )
) : BaseWebView(context), ButtonRecorder {

    init {
        webViewClient = webViewClientImpl

        scrollBarStyle = SCROLLBARS_INSIDE_OVERLAY
        isHorizontalScrollBarEnabled = false
        isVerticalScrollBarEnabled = false

        with(settings) {
            setSupportZoom(false)
            javaScriptEnabled = true
            domStorageEnabled = true
            allowFileAccess = false
            allowContentAccess = false
        }

        setBackgroundColor(Color.TRANSPARENT)

        visibility = GONE
    }

    val unrecoverableError = webViewClientImpl.unrecoverableError
    val clickthroughEvent: SharedFlow<Unit> = webViewClientImpl.clickthroughEvent
    val isPageFinished: StateFlow<Boolean> = webViewClientImpl.isPageFinished

    override fun destroy() {
        super.destroy()
        scope.cancel()
    }

    // Touch Listener that tracks the last click event
    @SuppressLint("ClickableViewAccessibility")
    override fun onTouchEvent(event: MotionEvent): Boolean {
        // blocks clickthrough if we are running the banner experiment
        // since the touch is intercepted and fired at the top layer banner
        if (isBannerClickthroughExperiment) return false

        if (event.action == MotionEvent.ACTION_DOWN) {
            val position = IntArray(2)
            this.getLocationOnScreen(position)
            webViewClientImpl.onLastTouch(
                AdTouch(
                    viewPositionX = position.first(),
                    viewPositionY = position.last(),
                    viewSizeHeight = this.height,
                    viewSizeWidth = this.width,
                    touchX = (event.x + position.first()).toInt(),
                    touchY = (event.y + position.last()).toInt()
                )
            )
        }
        return super.onTouchEvent(event)
    }

    // TODO. Refactor.
    // Currently it's single-time use, so no worries regarding isLoaded flag volatility.
    suspend fun loadHtml(
        html: String,
        timeout: Duration
    ): Result<Unit, StaticAdError> = withContext(DispatcherProvider().main) {
        try {
            loadDataWithDefaultBaseUrl(htmlCssFixer.applyCSSRenderingFix(html))
        } catch (e: Exception) {
            MolocoLogger.error(TAG, "loadHtml", e)
            return@withContext Result.Failure(StaticAdError.STATIC_AD_WEBVIEW_DATA_WITH_DEFAULT_BASE_URL_ERROR)
        }

        val isLoadOperationTimedOut = withTimeoutOrNull(timeout) {
            // Awaiting for web page either loading successfully or having an error.
            webViewClientImpl.isLoaded.combine(
                webViewClientImpl.unrecoverableError
            ) { isLoaded, unrecoverableError ->
                isLoaded to unrecoverableError
            }.first {
                val (isLoaded, unrecoverableError) = it
                isLoaded || unrecoverableError != null
            }
        } == null

        // load timeout: as per https://mlc.atlassian.net/browse/SDK-1751,
        // let's set static web ad loaded state to "true".
        //
        // Once the issue https://mlc.atlassian.net/browse/SDK-1751 resolved, consider returning "timeout" StaticWebViewLoadResult
        if (isLoadOperationTimedOut) {
            webViewClientImpl.enforceIsLoaded()
        }

        // Returning latest values back for handling.
        val isLoaded = webViewClientImpl.isLoaded.value
        val unrecoverableError = webViewClientImpl.unrecoverableError.value

        if (unrecoverableError != null) {
            Result.Failure(unrecoverableError)
        } else if (isLoaded) {
            Result.Success(Unit)
        } else {
            Result.Failure(StaticAdError.STATIC_AD_UNKNOWN_ERROR)
        }
    }

    override fun onButtonRendered(button: CustomUserEventBuilderService.UserInteraction.Button) {
        webViewClientImpl.onButtonRendered(button)
    }

    override fun onButtonUnRendered(
        buttonType: CustomUserEventBuilderService.UserInteraction.Button.ButtonType
    ) {
        webViewClientImpl.onButtonUnRendered(buttonType)
    }
}
