package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.creative.mraid

import android.graphics.Rect
import android.webkit.WebView
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidPlacementType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidScreenMetrics
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidViewState
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import org.json.JSONObject

internal interface MraidJsEventSender {
    fun sendMraidNativeCommandComplete(commandString: String)
    fun sendMraidError(commandString: String, msg: String)
    fun sendMraidIsViewable(isViewable: Boolean)
    fun sendMraidPlacementType(placementType: MraidPlacementType)
    fun sendMraidViewState(state: MraidViewState)
    fun sendMraidSupports(
        sms: Boolean,
        tel: Boolean,
        calendar: Boolean,
        storePicture: Boolean,
        inlineVideo: Boolean,
    )
    fun sendMraidScreenMetrics(screenMetrics: MraidScreenMetrics)
}


internal class MraidJsEventSenderImpl(private val webView: WebView): MraidJsEventSender {
    private fun sendJs(js: String) {
        // Must be on the same thread the webView was started in
        CoroutineScope(DispatcherProvider().main).launch {
            webView.loadUrl("javascript:$js")
        }
    }

    // tells the JS the native command was handled
    override fun sendMraidNativeCommandComplete(commandString: String) {
        val cmd = JSONObject.quote(commandString)
        sendJs("mraidbridge.nativeCallComplete($cmd)")
    }

    // tells the JS there were errors native message handling
    override fun sendMraidError(commandString: String, msg: String) {
        sendJs(
            "mraidbridge.notifyErrorEvent(${
                JSONObject.quote(
                commandString
            )}, ${JSONObject.quote(msg)})"
        )
    }

    override fun sendMraidIsViewable(isViewable: Boolean) {
        sendJs("mraidbridge.setIsViewable($isViewable)")
    }

    override fun sendMraidPlacementType(placementType: MraidPlacementType) {
        sendJs("mraidbridge.setPlacementType(${JSONObject.quote(placementType.value)})")
    }

    override fun sendMraidViewState(state: MraidViewState) {
        sendJs("mraidbridge.setState(${JSONObject.quote(state.value)})")
    }

    override fun sendMraidSupports(
        sms: Boolean,
        tel: Boolean,
        calendar: Boolean,
        storePicture: Boolean,
        inlineVideo: Boolean,
    ) {
        sendJs("mraidbridge.setSupports($sms,$tel,$calendar,$storePicture,$inlineVideo)")
    }

    override fun sendMraidScreenMetrics(screenMetrics: MraidScreenMetrics) {
        sendJs(
            """
                mraidbridge.setScreenSize(${stringifySize(screenMetrics.getScreenRectDips())});
                mraidbridge.setMaxSize(${stringifySize(screenMetrics.getRootViewRectDips())});
                mraidbridge.setCurrentPosition(${stringifyRect(screenMetrics.getCurrentAdRectDips())});
                mraidbridge.setDefaultPosition(${stringifyRect(screenMetrics.getDefaultAdRectDips())});
                mraidbridge.notifySizeChangeEvent(${stringifySize(screenMetrics.getCurrentAdRectDips())});
            """
        )
    }

    private fun stringifyRect(rect: Rect): String {
        return "${rect.left},${rect.top},${rect.width()},${rect.height()}"
    }

    private fun stringifySize(rect: Rect): String {
        return "${rect.width()},${rect.height()}"
    }
}
