package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Color
import android.view.ViewGroup
import android.webkit.WebView
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.common_adapter_internal.AdapterAccess.DispatcherProvider
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmResultTag
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidJsCommand.Close.toTemplateOrientationSettings
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.ad.orientation.AdOrientation
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.ad.orientation.OrientationSettings
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.creative.mraid.MraidCommunicationHub
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.errors.WebViewAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.PlayListItemDisplayingEventHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.RequiredContentEventHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeoutOrNull
import kotlin.time.Duration


@SuppressLint("SetJavaScriptEnabled", "ViewConstructor")
internal class TemplateWebView(
    context: Context,
    private val contentLoadedEventHandler: RequiredContentEventHandler,
    private val playListItemDisplayingEventHandler: PlayListItemDisplayingEventHandler,
    private val webViewClientImpl: TemplateWebViewClientImpl = TemplateWebViewClientImpl(
        contentLoadedEventHandler
    ),
) : WebView(context) {

    private val viewScope: CoroutineScope = CoroutineScope(DispatcherProvider().main)
    internal val orientation: StateFlow<OrientationSettings>
        get() = _orientationSettings
    private val _orientationSettings = MutableStateFlow(OrientationSettings(AdOrientation.None))

    init {
        webViewClient = webViewClientImpl

        scrollBarStyle = SCROLLBARS_INSIDE_OVERLAY
        isHorizontalScrollBarEnabled = false
        isVerticalScrollBarEnabled = false

        with(settings) {
            setSupportZoom(false)
            javaScriptEnabled = true
            domStorageEnabled = true
            mediaPlaybackRequiresUserGesture = false
            allowFileAccess = true
            allowContentAccess = true
        }

        setBackgroundColor(Color.TRANSPARENT)
    }

    val unrecoverableError = webViewClientImpl.unrecoverableError
    val isPageFinished: StateFlow<Boolean> = webViewClientImpl.isPageFinished

    /**
     * Starts collecting playlist item displaying events.
     */
    internal fun startCollectingPlaylistItemDisplaying(mraidCommunicationHub: MraidCommunicationHub) {
        viewScope.launch {
            try {
                playListItemDisplayingEventHandler.playlistItemDisplaying
                    .collect { creativeType ->
                        try {
                            withContext(Dispatchers.Main) {
                                settings.mediaPlaybackRequiresUserGesture = when(creativeType) {
                                    PlayListItemDisplayingEventHandler.PlaylistContainerType.MRAID -> true
                                    PlayListItemDisplayingEventHandler.PlaylistContainerType.VIDEO -> false
                                    else -> settings.mediaPlaybackRequiresUserGesture
                                }
                            }
                            MolocoLogger.info(TAG, "Set playback: ${settings.mediaPlaybackRequiresUserGesture}")
                            if (creativeType == PlayListItemDisplayingEventHandler.PlaylistContainerType.MRAID) {
                                MolocoLogger.info(TAG, "Playlist item displaying event is MRAID, setting orientation to: ${mraidCommunicationHub.expectedOrientation.value}")
                                _orientationSettings.value = mraidCommunicationHub.expectedOrientation.value.toTemplateOrientationSettings()
                                mraidCommunicationHub.sendMraidLoadEvents()
                            } else {
                                MolocoLogger.info(TAG, "Playlist item displaying event is not MRAID, setting orientation to none")
                                _orientationSettings.value = OrientationSettings(AdOrientation.None)
                            }
                        } catch (e: Exception) {
                            MolocoLogger.info(TAG, "Failed to access WebView settings", e)
                        }
                    }
            } catch (e: Exception) {
                MolocoLogger.info(TAG, "Error collecting playlist item displaying events", e)
            }
        }
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        viewScope.cancel()
    }

    // Currently it's single-time use, so no worries regarding isLoaded flag volatility.
    suspend fun loadHtml(
        html: String,
        timeout: Duration
    ): Result<Unit, WebViewAdError> =
        withContext(Dispatchers.Main) {
            val timerEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.WebviewLoadAd.eventName)
            try {
                loadDataWithBaseURL(null, html, "text/html", "UTF-8", null)
            } catch (e: Exception) {
                MolocoLogger.error(TAG, "loadHtml failed to load the provided html", e)
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewLoadAd.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, "invalid_url")
                )
                AndroidClientMetrics.recordTimerEvent(timerEvent
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, "invalid_url")
                )
                return@withContext Result.Failure(
                    WebViewAdError.WEBVIEW_DATA_WITH_DEFAULT_BASE_URL_ERROR
                )
            }

            val isLoadOperationTimedOut = withTimeoutOrNull(timeout) {
                // Awaiting for web page either loading successfully or having an error.
                webViewClientImpl.isLoaded.combine(
                    webViewClientImpl.unrecoverableError
                ) { isLoaded, unrecoverableError ->
                    isLoaded to unrecoverableError
                }.first {
                    val (isLoaded, unrecoverableError) = it
                    isLoaded || unrecoverableError != null
                }
            } == null

            if (isLoadOperationTimedOut) {
                MolocoLogger.error(TAG, "Ad failed to load due to timeout")
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewLoadAd.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, "timeout_error")
                )
                AndroidClientMetrics.recordTimerEvent(
                    timerEvent
                        .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                        .withTag(AcmTag.Reason.tagName, "timeout_error")
                )
                return@withContext Result.Failure(
                    WebViewAdError.WEBVIEW_TIMEOUT_ERROR
                )
            }

            // Returning latest values back for handling.
            val isLoaded = webViewClientImpl.isLoaded.value
            val unrecoverableError = webViewClientImpl.unrecoverableError.value

            if (unrecoverableError != null) {
                MolocoLogger.error(
                    TAG,
                    "Ad failed to load due to unrecoverable error: ${unrecoverableError.name}"
                )
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewLoadAd.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, unrecoverableError.name)
                )
                AndroidClientMetrics.recordTimerEvent(
                    timerEvent
                        .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                        .withTag(AcmTag.Reason.tagName, unrecoverableError.name)
                )
                Result.Failure(unrecoverableError)
            } else if (isLoaded) {
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewLoadAd.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.success.name)
                )
                AndroidClientMetrics.recordTimerEvent(timerEvent.withTag(AcmTag.Result.tagName, AcmResultTag.success.name))
                Result.Success(Unit)
            } else {
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewLoadAd.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, "unknown_error")
                )
                AndroidClientMetrics.recordTimerEvent(
                    timerEvent
                        .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                        .withTag(AcmTag.Reason.tagName, "unknown_error")
                )
                Result.Failure(WebViewAdError.UNKNOWN_ERROR)
            }
        }

    // This kind of helps avoiding WebView memory leaks.
    override fun destroy() {
        (parent as? ViewGroup)?.removeView(this)
        removeAllViews()
        super.destroy()
    }

    companion object {
        private const val TAG = "TemplateWebView"
    }
}
