package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer

import android.graphics.Bitmap
import android.webkit.RenderProcessGoneDetail
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebView
import androidx.webkit.WebResourceErrorCompat
import androidx.webkit.WebViewClientCompat
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.acm.TimerEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmResultTag
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.errors.WebViewAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.templates.renderer.events.handlers.RequiredContentEventHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.launch


internal class TemplateWebViewClientImpl(private val contentLoadedEventHandler: RequiredContentEventHandler) : WebViewClientCompat() {
    private val scope = CoroutineScope(DispatcherProvider().main)
    private val _isLoaded = MutableStateFlow(false)
    val isLoaded: StateFlow<Boolean> = _isLoaded

    private val _isPageFinished = MutableStateFlow(false)
    val isPageFinished = _isPageFinished.asStateFlow()

    private val _unrecoverableError = MutableStateFlow<WebViewAdError?>(null)
    val unrecoverableError = _unrecoverableError.asStateFlow()

    /**
     * This tracks the job that is used to determine if the error received is pending (for required or non-required asset)
     * If we are reloading the page, then we need to cancel the existing listener to avoid marking the page as finished multi
     */
    private var pendingErrorListener: Job? = null

    /**
     * Sometimes we get the following callback order:
     * onReceivedError -> onPageFinished -> view.evaluateJavascript
     *
     * And onPageFinished is called before the evaluateJavascript callback. This leads to a problem
     * where we determine page is successfully finished even when required assets could have failed
     */
    private val _hasPendingErrorResolution = MutableStateFlow(false)

    private var retryCount: Int = 0
    private val MAX_RETRY_LIMIT = 3

    private var requirecContentLoadListener: Job? = null
    private var pageLoadLatency: TimerEvent? = null
    private var requiredContentLoadLatency: TimerEvent? = null
    private var pageLoadError: String? = null

    private fun pageFinished() {
        MolocoLogger.info(TAG, "HTML Page finished loading is success: ${unrecoverableError.value == null}")
        _isPageFinished.value = true

        if (unrecoverableError.value != null) {
            MolocoLogger.info(TAG, "Unrecoverable error occurred, not setting isLoaded to true")
            _isLoaded.value = false
            requirecContentLoadListener?.cancel()

            AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewPageLoadEnd.eventName)
                .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                .withTag(AcmTag.Reason.tagName, pageLoadError ?: "unknown"))
            pageLoadLatency?.let {
                AndroidClientMetrics.recordTimerEvent(it
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag(AcmTag.Reason.tagName, pageLoadError ?: "unknown"))
            }

            return
        } else {
            MolocoLogger.info(TAG, "Waiting for content HTML assets to load or error out")

            AndroidClientMetrics.recordCountEvent(
                CountEvent(AcmCount.WebviewPageLoadEnd.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.success.name))
            pageLoadLatency?.let {
                AndroidClientMetrics.recordTimerEvent(it
                    .withTag(AcmTag.Result.tagName, AcmResultTag.success.name))
            }

            requirecContentLoadListener = scope.launch {
                contentLoadedEventHandler.contentLoadedEvent.collect { contentLoaded ->
                    MolocoLogger.info(TAG, "Content loaded event received, isSuccess: $contentLoaded")
                    _isLoaded.value = contentLoaded
                    requirecContentLoadListener?.cancel()

                    AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewRequiredContentLoaded.eventName)
                        .withTag(AcmTag.Result.tagName,
                            if (contentLoaded) AcmResultTag.success.name else AcmResultTag.failure.name))
                    requiredContentLoadLatency?.let {
                        AndroidClientMetrics.recordTimerEvent(it.withTag(
                            AcmTag.Result.tagName,
                            if (contentLoaded) AcmResultTag.success.name else AcmResultTag.failure.name
                        ))
                    }

                }
            }
        }
    }

    override fun onPageFinished(view: WebView?, url: String?) {
        super.onPageFinished(view, url)
        // Determine if the error received is pending (for required or non-required asset)
        MolocoLogger.info(TAG, "Webview page finished loading has pending error: ${_hasPendingErrorResolution.value}")
        AndroidClientMetrics.recordCountEvent(
            CountEvent(AcmCount.WebviewPageLoadFinishCallback.eventName)
                .withTag("pending_error", _hasPendingErrorResolution.value.toString()))
        if (!_hasPendingErrorResolution.value) {
            pageFinished()
        } else {
            pendingErrorListener = scope.launch {
                _hasPendingErrorResolution.collect { hasPendingError ->
                    MolocoLogger.info(TAG, "Webview page pending error resolution: $hasPendingError")
                    if (!hasPendingError) {
                        pageFinished()
                    }
                }
            }
        }
    }

    override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
        super.onPageStarted(view, url, favicon)
        MolocoLogger.info(TAG, "HTML Page started loading")
        AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewPageLoadStart.eventName))
        pageLoadLatency = AndroidClientMetrics.startTimerEvent(AcmTimer.WebviewPageLoadLatency.eventName)
        requiredContentLoadLatency = AndroidClientMetrics.startTimerEvent(AcmTimer.WebviewRequiredContentLoadLatency.eventName)
    }


    /**
     * Report an error to the host application. These errors are unrecoverable (i.e. the main resource is unavailable). The errorCode parameter corresponds to one of the ERROR_* constants.
     */
    // Looking for unrecoverable errors only, hence deprecated function is better for that (?)
    // This annotation is required by Kotlin in order to avoid warnings.
    @Suppress("DEPRECATION")
    @Deprecated("Deprecated in Android API 23")
    override fun onReceivedError(
        view: WebView,
        errorCode: Int,
        description: String,
        failingUrl: String
    ) {
        _hasPendingErrorResolution.value = true
        val url = failingUrl
        MolocoLogger.info(TAG, "Received error: ${errorCode}, with description: ${description} for url: ${url}")
        view.evaluateJavascript("ContentChecker.isRequiredContent('$url');") { isRequiredContentBooleanString ->
            val isRequired = isRequiredContentBooleanString.toBoolean()
            MolocoLogger.info(TAG, "[${Thread.currentThread().name}] Content type with webview error is required: $isRequired")
            if (isRequired) {
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewHtmlAdError.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag("required", "true")
                    .withTag(AcmTag.Reason.tagName, "${description}")
                    .withTag("status_code", "${errorCode}")
                    .withTag(AcmTag.RetryAttempt.tagName, "${retryCount}")
                    .withTag("is_loaded", "${isLoaded.value}")
                )
                if (shouldRetry(errorCode)) {
                    retryCount++
                    MolocoLogger.info(TAG, "Retrying... Attempt: ${retryCount}")
                    requirecContentLoadListener?.cancel()
                    view.reload()
                } else {
                    MolocoLogger.warn(TAG, "Retrying attempts complete. Setting unrecoverable error.")
                    pageLoadError = description
                    _unrecoverableError.value = WebViewAdError.WEBVIEW_RECEIVED_ERROR
                }
            } else {
                MolocoLogger.info(TAG, "Content is not required, not setting unrecoverable error")
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewHtmlAdError.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag("required", "false")
                    .withTag(AcmTag.Reason.tagName, description)
                    .withTag("status_code", "$errorCode")
                    .withTag("is_loaded", "${isLoaded.value}")
                )
            }
            _hasPendingErrorResolution.value = false
        }
        super.onReceivedError(view, errorCode, description, failingUrl)

    }

    /**
     * Report an error to the host application. These errors are unrecoverable (i.e. the main resource is unavailable). The errorCode parameter corresponds to one of the ERROR_* constants.
     */
    override fun onReceivedError(
        view: WebView,
        request: WebResourceRequest,
        error: WebResourceErrorCompat
    ) {
        _hasPendingErrorResolution.value = true
        MolocoLogger.info(TAG, "Received error: ${error.errorCode}, with description: ${error.description} for url: ${request.url}")
        val url = request.url.toString()
        view.evaluateJavascript("ContentChecker.isRequiredContent('$url');") { isRequiredContentBooleanString ->
            val isRequired = isRequiredContentBooleanString.toBoolean()
            MolocoLogger.info(TAG, "[${Thread.currentThread().name}] Content type with webview error is required: $isRequired")
            if (isRequired) {
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewHtmlAdError.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag("required", "true")
                    .withTag(AcmTag.Reason.tagName, "${error.description}")
                    .withTag("status_code", "${error.errorCode}")
                    .withTag(AcmTag.RetryAttempt.tagName, "${retryCount}")
                    .withTag("is_loaded", "${isLoaded.value}")
                )
                if (shouldRetry(error.errorCode)) {
                    retryCount++
                    MolocoLogger.info(TAG, "Retrying... Attempt: ${retryCount}")
                    requirecContentLoadListener?.cancel()
                    pendingErrorListener?.cancel()
                    view.reload()
                } else {
                    MolocoLogger.warn(TAG, "Retrying attempts complete. Setting unrecoverable error.")
                    pageLoadError = error.description.toString()
                    _unrecoverableError.value = WebViewAdError.WEBVIEW_RECEIVED_ERROR
                }
            } else {
                MolocoLogger.info(TAG, "Content is not required, not setting unrecoverable error")
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewHtmlAdError.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag("required", "false")
                    .withTag(AcmTag.Reason.tagName, "${error.description}")
                    .withTag("status_code", "${error.errorCode}")
                    .withTag("is_loaded", "${isLoaded.value}")
                )
            }
            _hasPendingErrorResolution.value = false
        }

        super.onReceivedError(view, request, error)
    }


    override fun onReceivedHttpError(
        view: WebView,
        request: WebResourceRequest,
        errorResponse: WebResourceResponse
    ) {
        _hasPendingErrorResolution.value = true
        MolocoLogger.info(TAG, "[${Thread.currentThread().name}] Received HTTP error: ${errorResponse?.statusCode}, with description: ${errorResponse?.reasonPhrase} for url: ${request?.url}")
        val url = request.url.toString()
        view.evaluateJavascript("ContentChecker.isRequiredContent('$url');") { isRequiredBooleanString ->
            val isRequired = isRequiredBooleanString.toBoolean()
            MolocoLogger.info(TAG, "[${Thread.currentThread().name}] Content isRequired with http error: $isRequired")
            if (isRequired) {
                _unrecoverableError.value = WebViewAdError.WEBVIEW_REQUIRED_CONTENT_HTTP_ERROR
                pageLoadError = errorResponse.statusCode.toString()
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewHtmlAdError.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag("required", "true")
                    .withTag(AcmTag.Reason.tagName, "${errorResponse.statusCode}")
                    .withTag("is_loaded", "${isLoaded.value}")
                )
                MolocoLogger.warn(TAG, "Setting unrecoverable error: ${unrecoverableError.value}")
            } else {
                MolocoLogger.info(TAG, "Content is not required, not setting unrecoverable error")
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewHtmlAdError.eventName)
                    .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
                    .withTag("required", "false")
                    .withTag(AcmTag.Reason.tagName, "${errorResponse.statusCode}")
                    .withTag("is_loaded", "${isLoaded.value}")
                )
            }
            _hasPendingErrorResolution.value = false
        }

        super.onReceivedHttpError(view, request, errorResponse)
    }

    override fun onRenderProcessGone(view: WebView, detail: RenderProcessGoneDetail?): Boolean {
        // https://developer.android.com/guide/webapps/managing-webview#termination-handle
        // Basically, then webview will be destroyed externally after this, which, ideally, isn't known here.
        // But who cares, plus deadlines.
        AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.WebviewHtmlAdError.eventName)
            .withTag(AcmTag.Result.tagName, AcmResultTag.failure.name)
            .withTag(AcmTag.Reason.tagName, "render_process_gone_error")
            .withTag("is_loaded", "${isLoaded.value}")
        )
        _unrecoverableError.value = WebViewAdError.WEBVIEW_RENDER_PROCESS_GONE_ERROR
        MolocoLogger.error(TAG, "onRenderProcessGone")
        _hasPendingErrorResolution.value = false
        return true
    }

    private fun shouldRetry(errorCode: Int): Boolean {
        // Retry should only happen if the error is either unknown or network error within the retry limit
        MolocoLogger.debug(TAG, "errorCode: $errorCode, (errorCode == UNKNOWN_ERROR || errorCode == NETWORK_ERROR): ${(errorCode == UNKNOWN_ERROR || errorCode == NETWORK_ERROR)}")
        MolocoLogger.debug(TAG, "retryCount: $retryCount, MAX_RETRY_LIMIT: $MAX_RETRY_LIMIT, retryCount < MAX_RETRY_LIMIT: ${retryCount < MAX_RETRY_LIMIT}")
        return (errorCode == UNKNOWN_ERROR || errorCode == NETWORK_ERROR) && retryCount < MAX_RETRY_LIMIT
    }

    companion object{
        private const val TAG = "TemplateWebViewClientImpl"

        private const val UNKNOWN_ERROR = -1
        private const val NETWORK_ERROR = -2
    }
}
