package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad

import android.content.Context
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidAdData
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidFullscreenContentController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.MraidFullscreenControllerEvent
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid.toPlaylist
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.VastError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.Ad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.GoNextAction
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.companion.CompanionController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.companion.CompanionControllerEvent
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.dec.DECController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.dec.DECControllerEvent
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.linear.LinearController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.linear.LinearControllerEvent
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.linear.NO_POSITION
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch

private const val TAG = "AdController"

/**
 * Returns [vastAdPlaylistController] instance for rendering VAST ads.
 * @param overrideLinearGoNextActionEnabled null: VAST linear skip/"go next" action availability relies on VAST xml ad markup's <Skippable> tag value; otherwise skip availability is decided by the Boolean value of this parameter.
 * @param overrideLinearGoNextActionEnabledDelaySeconds ignored when [overrideLinearGoNextActionEnabled] is null; otherwise skip action availability delay is decided by the Int value of this parameter.
 * @param companionGoNextActionDelaySeconds amount of seconds till "go next" action is available for Companion(endcard) ad parts.
 * @param decGoNextActionDelaySeconds amount of seconds till "go next" action is available for DEC ad parts.
 */
internal fun vastAdPlaylistController(
    ad: Ad,
    externalLinkHandler: ExternalLinkHandler,
    context: Context,
    customUserEventBuilderService: CustomUserEventBuilderService,
    mute: Boolean = true,
    overrideLinearGoNextActionEnabled: Boolean?,
    overrideLinearGoNextActionEnabledDelaySeconds: Int,
    companionGoNextActionDelaySeconds: Int,
    decGoNextActionDelaySeconds: Int,
    // Moloco feature: if user clicks skip button, open clicktrough link without VAST click tracking.
    autoStoreOnSkip: Boolean,
    // Moloco feature: if video ends, open clicktrough link without VAST click tracking.
    autoStoreOnComplete: Boolean
): AdController =
    AdPlaylistControllerImpl(
        playlist = ad.toPlaylist(
            externalLinkHandler,
            context,
            customUserEventBuilderService,
            mute,
            overrideLinearGoNextActionEnabled,
            overrideLinearGoNextActionEnabledDelaySeconds,
            companionGoNextActionDelaySeconds,
            decGoNextActionDelaySeconds,
            autoStoreOnSkip,
            autoStoreOnComplete
        ),
        adVastTracker = AdVastTracker(
            impressions = ad.impressionTracking,
            close = ad.linear.tracking.closeLinear,
            errors = ad.errorTracking
        )
    )


/**
 * Creates an instance of [AdController] using the given [MraidAdData] and related parameters.
 *
 * @param mraidAdData The MRAID ad data used to generate the playlist for the ad controller.
 * @param externalLinkHandler The handler for managing external links in the MRAID ad.
 * @param context The [Context] in which the ad controller operates, usually the activity or application context.
 * @param mraidFullscreenContentController An optional controller for handling fullscreen MRAID ads, may be `null`.
 * @param decGoNextActionDelaySeconds The delay in seconds before going to the next action in the playlist.
 * @param customUserEventBuilderService A service for building custom user events to be tracked during ad interactions.
 * @return An instance of [AdController] configured with the playlist created from the MRAID ad data.
 */
internal fun mraidAdPlaylistController(mraidAdData: MraidAdData,
                                       externalLinkHandler: ExternalLinkHandler,
                                       context: Context,
                                       mraidFullscreenContentController: MraidFullscreenContentController,
                                       decGoNextActionDelaySeconds: Int,
                                       customUserEventBuilderService: CustomUserEventBuilderService) : AdController =
    AdPlaylistControllerImpl(
        playlist = mraidAdData.toPlaylist(
            mraidFullscreenContentController,
            decGoNextActionDelaySeconds,
            context,
            externalLinkHandler,
            customUserEventBuilderService
        ),
        null
    )

internal class AdPlaylistControllerImpl(
    private val playlist: List<PlaylistItem>,
    private val adVastTracker: AdVastTracker?
) : AdController {

    private val scope = CoroutineScope(DispatcherProvider().main)
    private val currentPlaylistItem = MutableStateFlow<PlaylistItem?>(null)

    private fun tryGoToFirstPlaylistItem() {
        val firstPlaylistItem = playlist.firstOrNull()
            ?: return

        goToPlaylistItem(firstPlaylistItem)
    }

    private fun tryGoToNextPlaylistItem(): Boolean {
        val nextPlaylistItem = playlist.getOrNull(
            playlist.indexOf(currentPlaylistItem.value) + 1
        ) ?: return false

        goToPlaylistItem(nextPlaylistItem)
        return true
    }

    private fun goToPlaylistItem(playlistItem: PlaylistItem?) {
        currentPlaylistItem.value = playlistItem
        if (playlistItem is PlaylistItem.Linear) {
            playlistItem.linear.replay()
        }
    }

    override val currentAdPart: StateFlow<AdViewModel.AdPart?> = currentPlaylistItem.map { pi ->
        val isLastAdPart = playlist.lastOrNull() == pi

        when (pi) {
            is PlaylistItem.Companion -> AdViewModel.AdPart.Companion(pi.companion, isLastAdPart)
            is PlaylistItem.Linear -> AdViewModel.AdPart.Linear(pi.linear, isLastAdPart)
            is PlaylistItem.DEC -> AdViewModel.AdPart.DEC(pi.dec, isLastAdPart)
            is PlaylistItem.Mraid -> AdViewModel.AdPart.Mraid(pi.mraid, isLastAdPart)
            null -> null
        }
    }.stateIn(scope, SharingStarted.WhileSubscribed(), null)

    override val ctaAvailable = currentPlaylistItem.map {
        when (it) {
            is PlaylistItem.Companion -> it.companion.hasClickThrough
            is PlaylistItem.Linear -> it.linear.hasClickThrough
            is PlaylistItem.DEC -> false
            is PlaylistItem.Mraid -> false
            null -> false
        }
    }.stateIn(scope, SharingStarted.WhileSubscribed(), false)

    override fun onCTA() {
        when (val pi = currentPlaylistItem.value) {
            is PlaylistItem.Companion -> pi.companion.onClickThrough(NO_POSITION)
            is PlaylistItem.Linear -> pi.linear.onClickThrough(NO_POSITION)
            is PlaylistItem.DEC -> { MolocoLogger.warn(TAG, "Empty CTA DEC playlist item reached") }
            is PlaylistItem.Mraid -> { MolocoLogger.warn(TAG, "Empty CTA Mraid playlist item reached") }
            null -> { MolocoLogger.warn(TAG, "Empty CTA playlist item reached") }
        }
    }

    private val isCurrentLinearPlaylistItemPlaying =
        currentPlaylistItem.isLinearPlaylistItemPlayingFlow(scope)

    override val canReplay: StateFlow<Boolean> = combine(
        currentPlaylistItem,
        isCurrentLinearPlaylistItemPlaying
    ) { currentPlaylistItem, isCurrentLinearPlaylistItemPlaying ->

        currentPlaylistItem != null &&
            currentPlaylistItem == playlist.last() &&
            isCurrentLinearPlaylistItemPlaying != true
    }.stateIn(scope, SharingStarted.WhileSubscribed(), false)

    override fun onReplay() {
        tryGoToFirstPlaylistItem()
        onEvent(AdControllerEvent.Replay)
    }

    // Entry point into ad displaying.
    override fun show() {
        tryGoToFirstPlaylistItem()
    }

    private val adGoNextActionHolder = AdGoNextAction(
        currentPlaylistItem,
        scope
    )

    override val goNextAction by adGoNextActionHolder::goNextAction

    override fun goNextAdPartOrDismissAd() {
        if (goNextAction.value !is GoNextAction.State.Available) {
            return
        }

        // TODO. Common schema for linear, companion, dec navigation.
        val linearPlaylistItem = currentPlaylistItem.value as? PlaylistItem.Linear
        if (linearPlaylistItem != null) {
            linearPlaylistItem.linear.onSkip()
            return
        }

        tryGoToNextPlaylistItemOtherwiseCloseAd()
    }

    // Gets called ONLY for user initiated "go next" actions, that is, all skip/next button taps;
    // linear/video transition to the next item upon video completion is not included.
    private fun tryGoToNextPlaylistItemOtherwiseCloseAd() {
        // Quick workaround to track "skip/go to DEC" user initiated action.
        playlist.tryGetNextPlaylistItemAsDEC(currentPlaylistItem.value)
            ?.trackSkipToDEC()

        if (tryGoToNextPlaylistItem()) {
            return
        }

        // Close the ad when nothing is left to show.
        // It sends event to close from outside rather than closes itself internally.
        // Something to consider for refactoring.
        adVastTracker?.trackClose()
        onEvent(AdControllerEvent.Dismiss)
    }

    private val _event = MutableSharedFlow<AdControllerEvent>()
    override val event: Flow<AdControllerEvent> = _event

    private fun onEvent(event: AdControllerEvent) = scope.launch { _event.emit(event) }

    // start listening to all playlistitem's controllers right away so we don't overlook any events.
    init {
        for (pi in playlist) {
            when (pi) {
                is PlaylistItem.Companion -> pi.companion.event.onEach { event ->
                    when (event) {
                        is CompanionControllerEvent.Error -> {
                            adVastTracker?.trackError(VastError.Companion)
                            onEvent(AdControllerEvent.Error(event.error))
                        }
                        is CompanionControllerEvent.ClickThrough -> {
                            onEvent(AdControllerEvent.ClickThrough)
                        }
                        is CompanionControllerEvent.DisplayStarted -> {
                            adVastTracker?.trackImpression()
                            onEvent(AdControllerEvent.CompanionDisplayStarted)
                        }
                        is CompanionControllerEvent.DisplayEnded -> {
                            pi.companion.destroy()
                        }
                    }
                }
                is PlaylistItem.Linear -> pi.linear.event.onEach { event ->
                    when (event) {
                        is LinearControllerEvent.Error -> {
                            adVastTracker?.trackError(VastError.Linear)
                            onEvent(AdControllerEvent.Error(event.error))
                        }
                        LinearControllerEvent.ClickThrough -> {
                            onEvent(AdControllerEvent.ClickThrough)
                        }
                        LinearControllerEvent.Skip -> {
                            onEvent(AdControllerEvent.Skip)
                            tryGoToNextPlaylistItemOtherwiseCloseAd()
                        }
                        LinearControllerEvent.Complete -> {
                            onEvent(AdControllerEvent.Complete)
                            tryGoToNextPlaylistItem()
                        }
                        LinearControllerEvent.DisplayStarted -> {
                            adVastTracker?.trackImpression()
                            onEvent(AdControllerEvent.LinearDisplayStarted)
                        }
                    }
                }
                is PlaylistItem.DEC -> pi.dec.event.onEach { event ->
                    when (event) {
                        DECControllerEvent.ClickThrough -> {
                            onEvent(AdControllerEvent.ClickThrough)
                        }
                        DECControllerEvent.DisplayStarted -> {
                            adVastTracker?.trackImpression()
                            onEvent(AdControllerEvent.DECDisplayStarted)
                        }
                    }
                }
                is PlaylistItem.Mraid -> pi.mraid.event.onEach { event ->
                    when (event) {
                        MraidFullscreenControllerEvent.SkipOrClose -> {
                            onEvent(AdControllerEvent.Skip)
                            tryGoToNextPlaylistItemOtherwiseCloseAd()
                        }
                        MraidFullscreenControllerEvent.ClickThrough -> {
                            onEvent(AdControllerEvent.ClickThrough)
                        }
                    }
                }
            }.launchIn(scope)
        }
    }

    override fun destroy() {
        scope.cancel()

        playlist.onEach {
            when (it) {
                is PlaylistItem.Companion -> it.companion.destroy()
                is PlaylistItem.Linear -> it.linear.destroy()
                is PlaylistItem.DEC -> it.dec.destroy()
                is PlaylistItem.Mraid -> it.mraid.destroy()
            }
        }

        goToPlaylistItem(null)
    }

    override fun onButtonRendered(button: CustomUserEventBuilderService.UserInteraction.Button) {
        val validatedBtn = playlist.validateButton(
            currentPlaylistItem.value,
            button
        )

        when (val pi = currentPlaylistItem.value) {
            is PlaylistItem.Linear -> pi.linear.onButtonRendered(validatedBtn)
            is PlaylistItem.Companion -> pi.companion.onButtonRendered(validatedBtn)
            is PlaylistItem.DEC -> pi.dec.onButtonRendered(validatedBtn)
            is PlaylistItem.Mraid -> { MolocoLogger.warn(TAG, "Empty onButtonRendered MRAID playlist item reached") }
            null -> MolocoLogger.warn(
                TAG,
                "Displaying ${validatedBtn.buttonType} at position: ${validatedBtn.position} of size: ${validatedBtn.size} in unknown playlist item type"
            )
        }
    }

    override fun onButtonUnRendered(
        buttonType: CustomUserEventBuilderService.UserInteraction.Button.ButtonType
    ) {
        val validatedBtnType = playlist.validateButtonType(
            currentPlaylistItem.value,
            buttonType
        )

        when (val pi = currentPlaylistItem.value) {
            is PlaylistItem.Linear -> pi.linear.onButtonUnRendered(validatedBtnType)
            is PlaylistItem.Companion -> pi.companion.onButtonUnRendered(validatedBtnType)
            is PlaylistItem.DEC -> pi.dec.onButtonUnRendered(validatedBtnType)
            is PlaylistItem.Mraid -> { MolocoLogger.warn(TAG, "Empty onButtonUnRendered MRAID playlist item reached") }
            null -> MolocoLogger.warn(TAG, "Unrendering $validatedBtnType in unknown playlist item type")
        }
    }
}

internal sealed class PlaylistItem {
    class Linear(val linear: LinearController) : PlaylistItem()
    class Companion(val companion: CompanionController) : PlaylistItem()
    class DEC(val dec: DECController) : PlaylistItem()
    class Mraid(val mraid: MraidFullscreenContentController) : PlaylistItem()
}

// xxxLatest flow APIs are Experimental, that's why I use workarounds like that.
/**
 * @return _null_ - playlist item is __not__ [PlaylistItem.Linear]
 *
 * _boolean_ - playlist linear item's [LinearController.isPlaying] status
 */
private fun Flow<PlaylistItem?>.isLinearPlaylistItemPlayingFlow(
    scope: CoroutineScope
): Flow<Boolean?> {
    val isLinearPlayingFlow = MutableStateFlow<Boolean?>(null)

    val playlistItemFlow = this
    scope.launch {
        playlistItemFlow.collectLatest { pi ->
            when (pi) {
                // We're in collectLatest { ... }, previous subscriptions are automatically cancelled.
                is PlaylistItem.Linear -> {
                    pi.linear.isPlaying.collect {
                        isLinearPlayingFlow.value = it
                    }
                }
                else -> {
                    isLinearPlayingFlow.value = null
                }
            }
        }
    }

    return isLinearPlayingFlow
}

private fun List<PlaylistItem>.tryGetNextPlaylistItemAsDEC(
    currentItem: PlaylistItem?
): DECController? {
    val nextItem = getOrNull(indexOf(currentItem) + 1)
    return (nextItem as? PlaylistItem.DEC)?.dec
}

private fun List<PlaylistItem>.validateButtonType(
    forPlaylistItem: PlaylistItem?,
    buttonType: CustomUserEventBuilderService.UserInteraction.Button.ButtonType
): CustomUserEventBuilderService.UserInteraction.Button.ButtonType =
    // Quick workaround to support SKIP_DEC ButtonType without changing UI layer drastically:
    if (buttonType == CustomUserEventBuilderService.UserInteraction.Button.ButtonType.SKIP &&
        tryGetNextPlaylistItemAsDEC(forPlaylistItem) != null
    ) {
        CustomUserEventBuilderService.UserInteraction.Button.ButtonType.SKIP_DEC
    } else {
        buttonType
    }

private fun List<PlaylistItem>.validateButton(
    forPlaylistItem: PlaylistItem?,
    button: CustomUserEventBuilderService.UserInteraction.Button
): CustomUserEventBuilderService.UserInteraction.Button {
    val newButtonType = validateButtonType(
        forPlaylistItem,
        button.buttonType
    )

    return if (newButtonType == button.buttonType) {
        button
    } else {
        button.copy(buttonType = newButtonType)
    }
}
