package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.videoplayer

import android.view.View
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Destroyable
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.VastAdShowError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PlaybackProgress
import kotlinx.coroutines.flow.StateFlow

/**
 * Represents the state of the video player.
 * @param isPlaying true if the video is actively playing, false otherwise.
 * @param isVisible true if the video player is visible, false otherwise.
 * @param hasMore true if the player has more of the video content to play, false otherwise.
 *                This can be true with isPlaying false when the content is being streamed and not yet buffered
 */
data class PlayingState(val isPlaying: Boolean,
                        val isVisible: Boolean = true,
                        val hasMore: Boolean = true)


/**
 * Interface representing a video player with essential playback controls and state management.
 */
interface VideoPlayer : Destroyable {
    /**
     * The view associated with the video player. This may be `null` if the player is not initialized.
     */
    val playerView: View?

    /**
     * Controls whether the video is muted.
     */
    var isMute: Boolean

    /**
     * The URI source of the video to be played. Can be `null` if no video is set.
     */
    var uriSource: String?

    /**
     * Starts or resumes video playback.
     */
    fun play()

    /**
     * Seeks the video to the specified position.
     *
     * @param positionMillis The position to seek to, in milliseconds.
     */
    fun seekTo(positionMillis: Long)

    /**
     * Pauses the video playback.
     */
    fun pause()

    /**
     * Suspends until the video player is ready for playback.
     *
     * This method accounts for a known issue where ExoPlayer encounters difficulties playing H.265-encoded videos
     * on certain Pixel devices.
     */
    suspend fun awaitReady()

    /**
     * A [StateFlow] representing the current playing state of the video.
     */
    val isPlaying: StateFlow<PlayingState>

    /**
     * A [StateFlow] representing the playback progress of the video.
     */
    val playbackProgress: StateFlow<PlaybackProgress>

    /**
     * A [StateFlow] containing the last encountered error during video playback.
     */
    val lastError: StateFlow<VastAdShowError?>
}
