package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.webview

import android.content.Context
import android.util.Base64

/**
 * Interface for loading local assets from our resources.
 *
 */
internal interface LocalAssetLoader {
    /**
     * Loads the content of a file from local assets.
     *
     * @param fileName The name of the file to load from assets
     * @return The content of the file as a string
     */
    fun loadAsset(fileName: String): String
}

internal class AssetLoaderImpl(private val context: Context) : LocalAssetLoader {
    override fun loadAsset(fileName: String): String {
        return context.assets.open(fileName).bufferedReader().use { it.readText() }
    }
}

/**
 * Interface for encoding content into Base64 format.
 *
 */
internal interface ContentEncoder {
    /**
     * Encodes the provided content to Base64 format.
     *
     * @param content The string content to be encoded
     * @return Base64 encoded representation of the input content
     */
    fun encodeToBase64(content: String): String
}

internal class Base64EncoderImpl : ContentEncoder {
    override fun encodeToBase64(content: String): String {
        return Base64.encodeToString(
            content.trimIndent().toByteArray(),
            Base64.NO_PADDING
        )
    }
}

/**
 * Interface for injecting HTML content into iframes.
 *
 */
internal interface IframeInjector {
    /**
     * Injects the provided HTML content into an iframe with MRAID support.
     *
     * @param html The HTML content to be injected into the iframe
     * @return HTML string containing an iframe with the encoded content
     */
    fun injectIntoMraidIframe(html: String): String
}

internal class IframeInjectorImpl(
    private val localAssetLoader: LocalAssetLoader,
    private val contentEncoder: ContentEncoder,
    private val cssMarginHandler: HTMLCSSFixer = HTMLCSSFixer()
): IframeInjector {

    override fun injectIntoMraidIframe(html: String): String {
        val mraidJs = localAssetLoader.loadAsset("mraid.js")
        val mraidBridgeJs = localAssetLoader.loadAsset("mraid-bridge.js")

        val replaceMraidJSContent = cssMarginHandler.applyCSSRenderingFix(html).replace(
            "<script src=\"mraid.js\"></script>",
            "<script>$mraidJs</script>")

        val base64EncodedContent: String = contentEncoder.encodeToBase64(replaceMraidJSContent)

        return """
            <script>$mraidBridgeJs</script>
            <iframe id="adFrame"
                style="width:100vw; height:100vh; border:none;"
                src="data:text/html;base64,$base64EncodedContent"
                sandbox="allow-scripts allow-same-origin"
            >
            </iframe>
        """.trimIndent()
    }
}
