package com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport

import android.util.Patterns
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.core.requestTimeoutMillis
import io.ktor.client.HttpClient
import io.ktor.client.request.get
import io.ktor.client.request.headers
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import io.ktor.http.ContentType
import io.ktor.http.HttpHeaders
import io.ktor.http.HttpStatusCode
import io.ktor.http.contentType
import kotlinx.coroutines.withContext

private const val TAG = "HttpRequestClient"
internal const val TIMEOUT_MILLIS = 5000L
internal const val WORKER_MAXIMUM_ATTEMPT = 5

interface HttpRequestClient {
    fun send(url: String)
    fun sendPost(url: String, body: ByteArray, contentType: ContentType, contentEncoding: String? = null)
}

internal fun String.isValidUrl(): Boolean {
    return Patterns.WEB_URL.matcher(this).matches().also {
        if (!it) {
            MolocoLogger.error(TAG, "URL is invalid. $this")
        }
    }
}

internal suspend fun HttpClient.sendGetRequest(url: String) = try {
    withContext(DispatcherProvider().io) {
        val status = get(url) {
            requestTimeoutMillis(TIMEOUT_MILLIS)
        }.status

        status == HttpStatusCode.OK || status == HttpStatusCode.NoContent
    }
} catch (e: Exception) {
    false
}

internal suspend fun HttpClient.sendPostRequest(url: String, body: ByteArray, contentType: ContentType, contentEncoding: String? = null) = try {
    withContext(DispatcherProvider().io) {
        val status = post(url) {
            if (contentEncoding?.isNotBlank() == true) {
                headers {
                    append(HttpHeaders.ContentEncoding, contentEncoding)
                }
            }
            setBody(body)
            contentType(contentType)
            requestTimeoutMillis(TIMEOUT_MILLIS)
        }.status

        MolocoLogger.debugBuildLog("HttpClient", "Response status: $status for url: $url")

        status == HttpStatusCode.OK || status == HttpStatusCode.NoContent
    }
} catch (e: Exception) {
    false
}
