package com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport

import android.content.Context
import android.content.Context.CONNECTIVITY_SERVICE
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.Build
import androidx.annotation.RequiresApi
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.android_context.ApplicationContext
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.service_locator.SdkObjectFactory
import io.ktor.client.HttpClient
import io.ktor.http.ContentType
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

/**
 * A non persistent HTTP client that invokes GET on URLs. This request is not persisted across
 * app sessions and is not guaranteed to complete. Use this as fire and forget
 * Follows the same retry principles as [PersistentHttpRequest]
 */
private const val TAG = "NonPersistentRequest"

interface NonPersistentHttpRequest : HttpRequestClient

fun NonPersistentHttpRequest(): NonPersistentHttpRequest = Instance

private val Instance by lazy {
    NonPersistentHttpRequestImpl(SdkObjectFactory.Network.httpClientSingleton)
}

const val MIN_BACKOFF_MILLIS = 10 * 1000L // 10 seconds - Not referenced from workmanager library to avoid build conflicts

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal class NonPersistentHttpRequestImpl(
    private val httpClient: HttpClient,
) : NonPersistentHttpRequest {
    private val networkScope = CoroutineScope(DispatcherProvider().default)

    override fun send(url: String) {
        networkScope.launch {
            for (runAttemptCount in 0 until WORKER_MAXIMUM_ATTEMPT) {
                val networkAvailable = isNetworkAvailable(ApplicationContext())
                MolocoLogger.info(TAG, "Network available: $networkAvailable for non persistent request")
                val success = if (networkAvailable) httpClient.sendGetRequest(url) else false
                if (success) {
                    return@launch
                } else {
                    delay(MIN_BACKOFF_MILLIS)
                }
            }
        }
    }

    override fun sendPost(url: String, body: ByteArray, contentType: ContentType, contentEncoding: String?) {
        networkScope.launch {
            for (runAttemptCount in 0 until WORKER_MAXIMUM_ATTEMPT) {
                val networkAvailable = isNetworkAvailable(ApplicationContext())
                MolocoLogger.info(TAG, "Network available: $networkAvailable for non persistent request")
                val success = if (networkAvailable) httpClient.sendPostRequest(url, body, contentType) else false
                if (success) {
                    return@launch
                } else {
                    delay(MIN_BACKOFF_MILLIS)
                }
            }
        }
    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal fun isNetworkAvailable(context: Context): Boolean = with(context) {
        val connectivityManager = getSystemService(CONNECTIVITY_SERVICE) as ConnectivityManager

        return with(connectivityManager) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                isNetworkAvailableApi23Plus()
            } else {
                isNetworkAvailable()
            }
        }
    }

    @RequiresApi(Build.VERSION_CODES.M)
    private fun ConnectivityManager.isNetworkAvailableApi23Plus(): Boolean {
        val capabilities = getNetworkCapabilities(activeNetwork) ?: return false

        return with(capabilities) {
            hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) ||
                hasTransport(NetworkCapabilities.TRANSPORT_WIFI) ||
                hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET)
        }
    }

    @Suppress("DEPRECATION")
    private fun ConnectivityManager.isNetworkAvailable() =
        activeNetworkInfo?.isConnected ?: false
}
