package com.moloco.sdk.internal

import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.unit.DpSize
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.moloco.sdk.R
import com.moloco.sdk.internal.ortb.model.AutoStore
import com.moloco.sdk.internal.ortb.model.DEC
import com.moloco.sdk.internal.ortb.model.HorizontalAlignment
import com.moloco.sdk.internal.ortb.model.Mute
import com.moloco.sdk.internal.ortb.model.Player
import com.moloco.sdk.internal.ortb.model.ProgressBar
import com.moloco.sdk.internal.ortb.model.SkipClose
import com.moloco.sdk.internal.ortb.model.VerticalAlignment
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdWebViewOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AggregatedOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.VastOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.AdCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.CountdownButtonPart
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.adCloseAfterCountdownIcon
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.defaultAdCloseCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.webview.defaultAdWebViewRenderer
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.adSkipAfterCountdownIcon
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.defaultAdSkipCountdownButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.defaultMuteButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.defaultProgressBar
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.defaultReplayButton
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.defaultVastIcon
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.defaultVastRenderer

internal fun Player.toFullscreenAggregatedOptions(): AggregatedOptions {
    val banner = false

    val adWebViewOptions = toAdWebViewOptions(banner)

    return AggregatedOptions(
        vastOptions = toVastOptions(banner),
        mraidOptions = adWebViewOptions,
        staticOptions = adWebViewOptions
    )
}

internal fun DefaultMolocoFullscreenAdAggregatedOptions(): AggregatedOptions =
    DefaultPlayerExt.toFullscreenAggregatedOptions()

internal fun Player.toBannerAggregatedOptions(): AggregatedOptions {
    val banner = true

    val adWebViewOptions = toAdWebViewOptions(banner)

    return AggregatedOptions(
        vastOptions = toVastOptions(banner),
        mraidOptions = adWebViewOptions,
        staticOptions = adWebViewOptions
    )
}

internal fun DefaultMolocoBannerAggregatedOptions(): AggregatedOptions =
    DefaultPlayerExt.toBannerAggregatedOptions()

private fun Player.toVastOptions(banner: Boolean): VastOptions = VastOptions(
    startMuted = mute.mute,
    // if "player.skip" object is absent - do not override (default VAST <Linear skipoffset="xx:xx:xx"> based logic);
    // otherwise - enforce skip.
    overrideLinearGoNextActionEnabled = if (skip == null) null else true,
    overrideLinearGoNextActionEnabledDelaySeconds = skip?.delaySeconds?.toInt() ?: 0,
    autoStoreOnSkip = autoStore?.enabled == true && autoStore.onSkip,
    autoStoreOnComplete = autoStore?.enabled == true,
    companionGoNextActionDelaySeconds = close.delaySeconds.toInt(),
    decGoNextActionDelaySeconds = dec?.close?.delaySeconds?.toInt() ?: 0,
    VastRenderer = defaultVastRenderer(
        replayButton = {
            replay?.let {
                val size = it.controlSize?.toInt()?.dp?.let { cs -> DpSize(cs, cs) } ?: DefaultControlSize

                defaultReplayButton(
                    alignment = Alignment(it.horizontalAlignment, it.verticalAlignment),
                    padding = PaddingValues(it.padding.toInt().dp),
                    size = size,
                    iconSize = size * 0.65f,
                    color = it.foregroundColor,
                    backgroundColor = it.backgroundColor ?: DefaultBackgroundButtonColor,
                    icon = painterResource(id = R.drawable.moloco_replay)
                )
            }
        },
        muteButton = {
            val size = mute.controlSize?.toInt()?.dp?.let { cs -> DpSize(cs, cs) } ?: DefaultControlSize

            defaultMuteButton(
                alignment = Alignment(mute.horizontalAlignment, mute.verticalAlignment),
                padding = PaddingValues(mute.padding.toInt().dp),
                size = size,
                iconSize = size * 0.6f,
                color = mute.foregroundColor,
                backgroundColor = mute.backgroundColor ?: DefaultBackgroundButtonColor,
                muteIcon = painterResource(id = R.drawable.moloco_volume_off),
                unmuteIcon = painterResource(id = R.drawable.moloco_volume_on)
            )
        },
        adCloseCountdownButton = close.toCloseButton(banner),
        adSkipCountdownButton = {
            skip?.let {
                val size = it.controlSize.toInt().dp.let { size -> DpSize(size, size) }

                defaultAdSkipCountdownButton(
                    alignment = Alignment(it.horizontalAlignment, it.verticalAlignment),
                    padding = PaddingValues(it.padding.toInt().dp),
                    color = it.foregroundColor,
                    size = size,
                    fontSize = it.controlSize.toInt().sp / 2,
                    showCountdown = false,
                    afterCountdownButtonPart = defaultMolocoSkipAfterCountdownButtonPart(
                        size * 0.4f,
                        it.backgroundColor ?: DefaultBackgroundButtonColor
                    )
                )
            }
        },
        ctaButton = {
            if (banner) {
                // Do not show cta for banners.
                null
            } else {
                cta?.let {
                    molocoCTAButton(
                        alignment = Alignment(it.horizontalAlignment, it.verticalAlignment),
                        padding = PaddingValues(it.padding.toInt().dp),
                        text = it.text,
                        contentColor = it.foregroundColor,
                        backgroundColor = it.backgroundColor ?: MolocoBlue,
                        imageUri = it.imageUrl
                    )
                }
            }
        },
        overrideVastContainerOnClick = OverrideVastContainerOnClick(isAllAreaClickable),
        progressBar = {
            if (banner) {
                // Do not show cta for banners.
                null
            } else {
                progressBar?.let {
                    defaultProgressBar(
                        alignment = Alignment(it.horizontalAlignment, it.verticalAlignment),
                        padding = PaddingValues(it.padding.toInt().dp),
                        color = it.foregroundColor
                    )
                }
            }
        },
        vastIcon = {
            vastPrivacyIcon?.let {
                defaultVastIcon(
                    alignment = Alignment(
                        it.horizontalAlignment,
                        it.verticalAlignment
                    ),
                    padding = PaddingValues(it.padding.toInt().dp)
                )
            } ?: defaultVastIcon()
        }
    )
)

@Composable
private fun defaultMolocoSkipAfterCountdownButtonPart(
    size: DpSize,
    backgroundColor: Color,
): CountdownButtonPart = adSkipAfterCountdownIcon(
    painter = painterResource(id = R.drawable.moloco_skip),
    iconSize = size,
    backgroundColor = backgroundColor
)

// skip functionality will be ignored for webView mraid; only use close
private fun Player.toAdWebViewOptions(isBanner: Boolean): AdWebViewOptions = AdWebViewOptions(
    closeDelaySeconds = close.delaySeconds.toInt(), // we'll still use the close delay data to render skip
    AdWebViewRenderer = defaultAdWebViewRenderer(
        adCloseCountdownButton = determineSkipCloseBehaviorFromDec(isBanner, close, dec)
    ),
    decGoNextActionDelaySeconds = dec?.close?.delaySeconds?.toInt() ?: 0,
    decClose = close.toCloseButton(isBanner)
)

internal fun determineSkipCloseBehaviorFromDec(
    isBanner: Boolean, close: SkipClose, dec: DEC?
): @Composable () -> AdCountdownButton? {
    return if (dec == null) {
        close.toCloseButton(isBanner)
    } else {
        {
            val size = close.controlSize.toInt().dp.let { size -> DpSize(size, size) }
            defaultAdSkipCountdownButton(
                alignment = Alignment(close.horizontalAlignment, close.verticalAlignment),
                padding = PaddingValues(close.padding.toInt().dp),
                color = close.foregroundColor,
                size = size,
                fontSize = close.controlSize.toInt().sp / 2,
                showCountdown = false,
                afterCountdownButtonPart = defaultMolocoSkipAfterCountdownButtonPart(
                    size * 0.4f,
                    close.backgroundColor ?: DefaultBackgroundButtonColor
                )
            )
        }
    }
}

private fun SkipClose.toCloseButton(banner: Boolean): @Composable () -> AdCountdownButton? = {
    if (banner) {
        // Do not show button for banners.
        null
    } else {
        val size = controlSize.toInt().dp.let { DpSize(it, it) }

        defaultAdCloseCountdownButton(
            alignment = Alignment(horizontalAlignment, verticalAlignment),
            padding = PaddingValues(padding.toInt().dp),
            color = foregroundColor,
            size = size,
            fontSize = controlSize.toInt().sp / 2,
            showCountdown = false,
            afterCountdownButtonPart = adCloseAfterCountdownIcon(
                painter = painterResource(id = R.drawable.moloco_close),
                iconSize = size * 0.45f,
                backgroundColor = backgroundColor ?: DefaultBackgroundButtonColor
            )
        )
    }
}

private fun Alignment(
    horizontalAlignment: HorizontalAlignment,
    verticalAlignment: VerticalAlignment,
): Alignment = when {
    verticalAlignment == VerticalAlignment.Top && (horizontalAlignment == HorizontalAlignment.Start || horizontalAlignment == HorizontalAlignment.Left) -> Alignment.TopStart
    verticalAlignment == VerticalAlignment.Top && horizontalAlignment == HorizontalAlignment.Center -> Alignment.TopCenter
    verticalAlignment == VerticalAlignment.Top && (horizontalAlignment == HorizontalAlignment.End || horizontalAlignment == HorizontalAlignment.Right) -> Alignment.TopEnd
    verticalAlignment == VerticalAlignment.Center && (horizontalAlignment == HorizontalAlignment.Start || horizontalAlignment == HorizontalAlignment.Left) -> Alignment.CenterStart
    verticalAlignment == VerticalAlignment.Center && horizontalAlignment == HorizontalAlignment.Center -> Alignment.Center
    verticalAlignment == VerticalAlignment.Center && (horizontalAlignment == HorizontalAlignment.End || horizontalAlignment == HorizontalAlignment.Right) -> Alignment.CenterEnd
    verticalAlignment == VerticalAlignment.Bottom && (horizontalAlignment == HorizontalAlignment.Start || horizontalAlignment == HorizontalAlignment.Left) -> Alignment.BottomStart
    verticalAlignment == VerticalAlignment.Bottom && horizontalAlignment == HorizontalAlignment.Center -> Alignment.BottomCenter
    verticalAlignment == VerticalAlignment.Bottom && (horizontalAlignment == HorizontalAlignment.End || horizontalAlignment == HorizontalAlignment.Right) -> Alignment.BottomEnd
    else -> Alignment.TopStart
}

private val DefaultPlayerExt: Player by lazy {
    val padding = 10u
    val controlSize = DefaultControlSizeRaw
    val foregroundColor = DefaultForegroundColor

    val skipClose = SkipClose(
        delaySeconds = 5u,
        padding = padding,
        controlSize = controlSize.toUInt(),
        horizontalAlignment = HorizontalAlignment.End,
        verticalAlignment = VerticalAlignment.Top,
        foregroundColor = foregroundColor
    )

    Player(
        skip = skipClose,
        close = skipClose,
        progressBar = ProgressBar(
            padding = 0u,
            horizontalAlignment = HorizontalAlignment.Center,
            verticalAlignment = VerticalAlignment.Bottom,
            foregroundColor = DefaultForegroundColor
        ),
        mute = Mute(
            mute = false,
            padding = padding,
            horizontalAlignment = HorizontalAlignment.Start,
            verticalAlignment = VerticalAlignment.Top,
            foregroundColor = foregroundColor
        ),
        replay = null,
        cta = null,
        autoStore = AutoStore(
            enabled = false
        ),
        isAllAreaClickable = true
    )
}

private val DefaultBackgroundButtonColor = Color.White
private val DefaultForegroundColor = MolocoBlue
private const val DefaultControlSizeRaw = 30
private val DefaultControlSize = DpSize(DefaultControlSizeRaw.dp, DefaultControlSizeRaw.dp)
