package com.moloco.sdk.internal.error.crash

import com.moloco.sdk.acm.AndroidClientMetrics as acm
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.error.api.ErrorReportingApi
import com.moloco.sdk.internal.error.crash.filters.ExceptionFilter

/**
 * Service that can handle crashes based on filters and report them
 */
internal interface CrashHandlerService {
    fun handleCrash(crash: Throwable)
}

internal class CrashHandlerServiceImpl(
    private val exceptionFilters: List<ExceptionFilter>,
    private val errorReporter: ErrorReportingApi,
) : CrashHandlerService {
    private val TAG = "CrashHandlerService"
    override fun handleCrash(crash: Throwable) {
        if (detectSDKCrashed(crash)) {
            acm.recordCountEvent(CountEvent(AcmCount.CrashDetected.eventName))
            errorReporter.reportFatalError(crash)
        } else {
            MolocoLogger.error(TAG, "App Crashed")
        }
    }

    private fun detectSDKCrashed(crash: Throwable): Boolean {
        // check if moloco SDK crashed by checking for packageName in the throwable stacktrace
        // if it did, return true
        // else return false
        // We look at cause because the crash can be wrapped in another exception
        for (filter in exceptionFilters) {
            if (filter.apply(crash)) {
                return true
            } else {
                // NO OP
            }
        }

        return false
    }
}
