package com.moloco.sdk.internal.ortb.model

import androidx.compose.ui.graphics.Color
import androidx.core.graphics.toColorInt
import kotlinx.serialization.KSerializer
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.descriptors.PrimitiveKind
import kotlinx.serialization.descriptors.PrimitiveSerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder

@Serializable
internal class Player(
    // Skip.
    // null - functionality is unavailable (control is hidden).
    @SerialName("skip") val skip: SkipClose? = null,
    // Close customization.
    @SerialName("close") val close: SkipClose,
    // Progress bar customization.
    // null - functionality is unavailable (control is hidden).
    @SerialName("progress_bar") val progressBar: ProgressBar? = null,
    // VAST player Mute customization.
    @SerialName("mute") val mute: Mute,
    // Replay Control customization.
    // null - functionality is unavailable (control is hidden).
    @SerialName("replay") val replay: Replay? = null,
    // CTA (Call to action) Control customization.
    @SerialName("cta") val cta: CTA? = null,
    // true - all video player / image endcard area is clickable, false - only CTA button.
    @SerialName("is_all_area_clickable") val isAllAreaClickable: Boolean,
    @SerialName("auto_store") val autoStore: AutoStore? = null,
    // Vast privacy icon customization.
    @SerialName("vast_privacy_icon") val vastPrivacyIcon: VastPrivacyIcon? = null,
    // Absence of dec would indicate to not show DEC (double endcard)
    @SerialName("dec") val dec: DEC? = null
)

// Schema https://docs.google.com/document/d/1NriLT8NNX6MLgRt-PxYLCPzyj-2gL2Y8qAzAZuiBOqM/edit#heading=h.93erlcpzjw6e
@Serializable
internal class DEC(
    // Advertising's icon image URL (cdn link)
    @SerialName("app_icon_url") val appIconUrl: String? = null,
    // Advertising app name.
    @SerialName("app_name") val appName: String? = null,
    // Link to fire upon impression on DEC.
    @SerialName("imp_link") val impressionTrackingUrl: String? = null,
    // click link to redirect the user to (could be app store URL or moloco tracking URL).
    @SerialName("click_through") val ctaUrl: String? = null,
    // click link for click tracking (only Moloco URL only if "click_through" field is filled with app store URL.
    @SerialName("click_tracking") val ctaTrackingUrl: String? = null,
    // Customizable CTA text to be displayed on DEC.
    @SerialName("cta_text") val ctaText: String? = null,
    // notifnt event link for notifying the skip that got to DEC.
    @SerialName("skip_event") val skipToDecTrackingUrl: String? = null,
    @SerialName("close") val close: DECSkipClose? = null
)

// TODO. replace with SkipClose once back-end ready and ready to support full DEC button customization.
@Serializable
internal class DECSkipClose(
    // Amount of seconds till Skip/Close Control is available; used in Skip/Close countdown timer.
    @SerialName("delay_seconds") val delaySeconds: UInt? = null
)

@Serializable
internal class SkipClose(
    // Amount of seconds till Skip/Close Control is available; used in Skip/Close countdown timer.
    @SerialName("delay_seconds") val delaySeconds: UInt,
    // padding in density independent pixels.
    @SerialName("padding") val padding: UInt,
    // Control size in dependency independent pixels.
    @SerialName("control_size") val controlSize: UInt,
    // Control alignment within the player area.
    @SerialName("horizontal_alignment") val horizontalAlignment: HorizontalAlignment,
    @SerialName("vertical_alignment") val verticalAlignment: VerticalAlignment,
    // Foreground/accent color of the Control.
    @Serializable(ColorSerializer::class)
    @SerialName("foreground_color")
    val foregroundColor: Color,

    // TODO. Remove null default value, once schema is stabilized.
    @Serializable(ColorSerializer::class)
    @SerialName("background_color")
    val backgroundColor: Color? = null
)

@Serializable
internal class ProgressBar(
    // padding in density independent pixels.
    @SerialName("padding") val padding: UInt,
    @SerialName("horizontal_alignment") val horizontalAlignment: HorizontalAlignment,
    @SerialName("vertical_alignment") val verticalAlignment: VerticalAlignment,
    // Foreground/accent color of the Control.
    @Serializable(ColorSerializer::class)
    @SerialName("foreground_color")
    val foregroundColor: Color
)

@Serializable
internal class Mute(
    // true - VAST video player starts with volume off.
    @SerialName("mute") val mute: Boolean,
    // padding in density independent pixels.
    @SerialName("padding") val padding: UInt,
    @SerialName("horizontal_alignment") val horizontalAlignment: HorizontalAlignment,
    @SerialName("vertical_alignment") val verticalAlignment: VerticalAlignment,
    // Foreground/accent color of the Control.
    @Serializable(ColorSerializer::class)
    @SerialName("foreground_color")
    val foregroundColor: Color,

    // TODO. Remove null default value, once schema is stabilized.
    // Control size in dependency independent pixels.
    @SerialName("control_size")
    val controlSize: UInt? = null,
    @Serializable(ColorSerializer::class)
    @SerialName("background_color")
    val backgroundColor: Color? = null
)

@Serializable
internal class Replay(
    // padding in density independent pixels.
    @SerialName("padding") val padding: UInt,
    @SerialName("horizontal_alignment") val horizontalAlignment: HorizontalAlignment,
    @SerialName("vertical_alignment") val verticalAlignment: VerticalAlignment,
    // Foreground/accent color of the Control.
    @Serializable(ColorSerializer::class)
    @SerialName("foreground_color")
    val foregroundColor: Color,

    // TODO. Remove null default value, once schema is stabilized.
    // Control size in dependency independent pixels.
    @SerialName("control_size")
    val controlSize: UInt? = null,
    @Serializable(ColorSerializer::class)
    @SerialName("background_color")
    val backgroundColor: Color? = null
)

@Serializable
internal class CTA(
    // Text to show on the CTA Control, like "Learn More", "Install Now" etc.
    @SerialName("text") val text: String,
    // Image/game icon etc to display along with text if available.
    // null - no image provided.
    @SerialName("image_url") val imageUrl: String? = null,
    // padding in density independent pixels.
    @SerialName("padding") val padding: UInt,
    @SerialName("horizontal_alignment") val horizontalAlignment: HorizontalAlignment,
    @SerialName("vertical_alignment") val verticalAlignment: VerticalAlignment,
    // Foreground/accent color of the Control, text color in this case.
    @Serializable(ColorSerializer::class)
    @SerialName("foreground_color")
    val foregroundColor: Color,

    // TODO. Remove null default value, once schema is stabilized.
    @Serializable(ColorSerializer::class)
    @SerialName("background_color")
    val backgroundColor: Color? = null
)

@Serializable
internal class AutoStore(
    // When "enabled" field is TRUE and User SKIPS:
    // handle VAST <ClickThrough> link as a general User CLICK (open app store, redirect to a website),
    // but DO NOT track VAST <ClickTracking> links in this case
    @SerialName("enabled") val enabled: Boolean,
    // onSkip == true Autostore should pop up once either on skip or before endcard if user does.
    // onSkip == false Autostore should pop up before endcard if user does not skip.
    @SerialName("on_skip") val onSkip: Boolean = true,
    // when "enabled" field is TRUE and "eventLink" not null -
    // request HTTPS event link to track the "autoClickOnSkip" event.
    @SerialName("event_link") val eventLink: String? = null
)

@Serializable
internal class VastPrivacyIcon(
    // padding in density independent pixels.
    @SerialName("padding") val padding: UInt,
    @SerialName("horizontal_alignment") val horizontalAlignment: HorizontalAlignment,
    @SerialName("vertical_alignment") val verticalAlignment: VerticalAlignment
)

@Serializable
internal enum class HorizontalAlignment {
    @SerialName("start")
    Start,

    @SerialName("center")
    Center,

    @SerialName("end")
    End,

    @SerialName("left")
    Left,

    @SerialName("right")
    Right
}

@Serializable
internal enum class VerticalAlignment {
    @SerialName("top")
    Top,

    @SerialName("center")
    Center,

    @SerialName("bottom")
    Bottom
}

private object ColorSerializer : KSerializer<Color> {
    override val descriptor = PrimitiveSerialDescriptor("Color", PrimitiveKind.STRING)

    override fun deserialize(decoder: Decoder): Color = Color(decoder.decodeString().toColorInt())

    override fun serialize(encoder: Encoder, value: Color) {
        // No need to serialize Color.
        throw NotImplementedError("Color encoding is not supported")
    }
}
