package com.moloco.sdk.internal.publisher

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.InternalSDKErrorSubType
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.createInternalAdErrorInfo
import com.moloco.sdk.internal.ortb.BidResponseParser
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.ortb.model.BidResponse
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.publisher.AdFormatType
import com.moloco.sdk.publisher.AdLoad
import com.moloco.sdk.publisher.MolocoAdError.ErrorType
import com.moloco.sdk.publisher.createAdErrorInfo
import com.moloco.sdk.publisher.createAdInfo
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.AdLoadTimeoutError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.ensureActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus
import kotlin.time.Duration
import com.moloco.sdk.acm.AndroidClientMetrics as acm

internal fun AdLoad(
        scope: CoroutineScope,
        timeout: (adLoadStartTimeMs: Long) -> Duration,
        adUnitId: String,
        // Once AdRenderer::ALoad::load() supports adm parameter, I won't need this.
        recreateXenossAdLoader: (bid: Bid) -> com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad,
        adFormatType: AdFormatType,
): AdLoad =
    AdLoadImpl(
        scope,
        timeout,
        adUnitId,
        recreateXenossAdLoader,
        BidResponseParser(),
        adLoadPreprocessors(),
        adFormatType,
    )

@VisibleForTesting
internal class AdLoadImpl(
        scope: CoroutineScope,
        private val timeout: (adLoadStartTimeMs: Long) -> Duration,
        // Once AdRenderer::ALoad::load() supports adm parameter, I won't need this.
        private val adUnitId: String,
        private val recreateXenossAdLoader: (
                bid: Bid,
        ) -> com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad,
        private val parseBidResponse: BidResponseParser,
        private val adLoadPreprocessors: List<AdLoadPreprocessor>,
        private val adFormatType: AdFormatType,
) : AdLoad {

    // Will automatically cancel when the parent scope is cancelled.
    private val scope = scope + DispatcherProvider().main

    override var isLoaded: Boolean = false

    private var currentBidResponseJson: String? = null
    private var currentBidResponse: BidResponse? = null
    private val acmLoadTimerEvent = acm.startTimerEvent(AcmTimer.LoadAd.eventName)

    private val BidResponse?.bid: Bid?
        get() = this?.seatBid?.get(0)?.bid?.get(0)

    override fun load(bidResponseJson: String, listener: AdLoad.Listener?) {
        val adLoadStartTimeMs = SdkObjectFactory.Miscellaneous.timeProviderSingleton()
        MolocoLogger.debug("AdLoadImpl", "load() called with bidResponseJson: $bidResponseJson")
        acmLoadTimerEvent.startTimer()
        acm.recordCountEvent(
            CountEvent(AcmCount.LoadAdAttempt.eventName)
                .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
        )

        // Wrapping by launch { ... } block to make sure the fun call is executed on the main thread.
        scope.launch {
            val processedBidResponseJson = processBidResponse(bidResponseJson)

            if (processedBidResponseJson == null) {
                MolocoLogger.warn(
                    "AdLoadImpl",
                    "Could not pre-process the bid response. Failing the load() call."
                )
                listener?.onAdLoadFailed(
                    createAdErrorInfo(adUnitId, ErrorType.AD_BID_PARSE_ERROR)
                )

                acm.recordTimerEvent(
                    acmLoadTimerEvent
                        .withTag(AcmTag.Result.tagName, "failure")
                        .withTag(AcmTag.Reason.tagName, "${ErrorType.AD_BID_PARSE_ERROR.errorCode}")
                        .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
                )
                acm.recordCountEvent(
                    CountEvent(AcmCount.LoadAdFailed.eventName)
                        .withTag(AcmTag.Reason.tagName, "${ErrorType.AD_BID_PARSE_ERROR.errorCode}")
                        .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
                )

                return@launch
            }

            MolocoLogger.info(
                "AdLoadImpl",
                "Processed the bidResponse, proceeding with the load() call."
            )

            // Wrapping the listener with tracking.
            val listenerTracker = AdLoadListenerTracker(
                originListener = listener,
                provideSdkEvents = { currentBidResponse?.bid?.ext?.sdkEvents },
                acmLoadTimerEvent = acmLoadTimerEvent,
                adFormatType = adFormatType
            )

            // If already loaded or in the process of loading of the same ad - ignore load call.
            if (currentBidResponseJson == processedBidResponseJson) {
                if (isLoaded) {
                    val adInfo = createAdInfo(adUnitId)
                    listenerTracker.onAdLoadStarted(adInfo, adLoadStartTimeMs)
                    listenerTracker.onAdLoadSuccess(adInfo)
                    return@launch
                }
                if (loadJob?.isActive == true) {
                    return@launch
                }
            }

            // Otherwise start load operation.
            startLoadJob(processedBidResponseJson, adLoadStartTimeMs, listenerTracker)
        }
    }

    /**
     * Pre-processes the bid response for the current mediation. If the mediation is not recognized,
     * the original bid response is returned.
     */
    private suspend fun processBidResponse(bidResponseJson: String): String? {
        for (preprocessor in adLoadPreprocessors) {
            if (preprocessor.canProcess()) {
                return preprocessor.process(bidResponseJson)
            }
        }
        return bidResponseJson
    }

    private var loadJob: Job? = null

    private fun startLoadJob(bidResponseJson: String, adLoadStartTimeMs: Long, listener: InternalAdLoadListener) {
        loadJob?.cancel()
        loadJob = scope.launch {
            isLoaded = false

            // Resetting cached bid response from the previous load() call.
            if (currentBidResponseJson != bidResponseJson) {
                currentBidResponseJson = bidResponseJson
                currentBidResponse = null
            }

            val bidResponse = currentBidResponse ?: parseBidResponse(bidResponseJson).let { res ->
                // Don't go further if an another load() call is in queue.
                ensureActive()

                (res as? Result.Success)?.value.also {
                    currentBidResponse = it
                    // Current bid response has to be set before listener APIs can be invoked
                    // Only report onAdLoadStarted if it's a new bid
                    listener.onAdLoadStarted(createAdInfo(adUnitId), adLoadStartTimeMs)
                }
            }

            val bid = bidResponse?.bid

            val adm = bid?.adm
            if (adm == null) {
                listener.onAdLoadFailed(createInternalAdErrorInfo(adUnitId, ErrorType.AD_BID_PARSE_ERROR, InternalSDKErrorSubType.AD_LOAD_BID_PARSE_ERROR_ADM_IS_NULL))
                return@launch
            }

            // Once recreateXenossAdLoader() is called
            // I expect an old loader to cancel all potential load() related callbacks.
            recreateXenossAdLoader(bid).load(
                timeout = timeout(adLoadStartTimeMs),
                listener = object : com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad.Listener {
                    override fun onLoad() {
                        scope.launch {
                            isLoaded = true
                            listener.onAdLoadSuccess(
                                createAdInfo(adUnitId, bid.price)
                            )
                        }
                    }

                    override fun onLoadTimeout(timeoutError: AdLoadTimeoutError) {
                        scope.launch {
                            isLoaded = false
                            listener.onAdLoadFailed(
                                createInternalAdErrorInfo(adUnitId, ErrorType.AD_LOAD_TIMEOUT_ERROR, timeoutError)
                            )
                        }
                    }

                    override fun onLoadError(internalError: MolocoAdSubErrorType) {
                        scope.launch {
                            isLoaded = false

                            listener.onAdLoadFailed(
                                createInternalAdErrorInfo(adUnitId, ErrorType.AD_LOAD_FAILED, internalError)
                            )
                        }
                    }
                }
            )
        }
    }
}
