package com.moloco.sdk.internal.publisher

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.acm.AndroidClientMetrics as acm
import com.moloco.sdk.acm.TimerEvent
import com.moloco.sdk.internal.MolocoInternalAdError
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.SdkEventUrlTracker
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.ortb.model.SdkEvents
import com.moloco.sdk.publisher.AdFormatType
import com.moloco.sdk.publisher.AdLoad
import com.moloco.sdk.publisher.MolocoAd

/**
 * Wraps the origin listener and fires url tracking event upon calling the respective listener's callback.
 */
internal fun AdLoadListenerTracker(
    originListener: AdLoad.Listener?,
    provideSdkEvents: () -> SdkEvents?,
    acmLoadTimerEvent: TimerEvent,
    adFormatType: AdFormatType
): InternalAdLoadListener = AdLoadListenerTrackerImpl(
    originListener,
    provideSdkEvents,
    SdkEventUrlTracker(),
    acmLoadTimerEvent,
    adFormatType
)

@VisibleForTesting
internal class AdLoadListenerTrackerImpl(
    private val originListener: AdLoad.Listener?,
    private val provideSdkEvents: () -> SdkEvents?,
    private val sdkEventUrlTracker: SdkEventUrlTracker,
    private val acmLoadTimerEvent: TimerEvent,
    private val adFormatType: AdFormatType
) : InternalAdLoadListener {
    private val TAG = "AdLoadListenerTrackerImpl"
    override fun onAdLoadStarted(molocoAd: MolocoAd, timestamp: Long) {
        MolocoLogger.info(TAG, "onAdLoadStarted: $molocoAd, $timestamp")
        provideSdkEvents()?.onAdLoadStart?.let { url ->
            sdkEventUrlTracker(url, timestamp)
        }
    }

    override fun onAdLoadSuccess(molocoAd: MolocoAd) {
        MolocoLogger.info(TAG, "onAdLoadSuccess: $molocoAd")
        provideSdkEvents()?.onAdLoadSuccess?.let { url ->
            sdkEventUrlTracker(url, System.currentTimeMillis())
        }

        acm.recordTimerEvent(
            acmLoadTimerEvent
                .withTag(AcmTag.Result.tagName, "success")
                .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
        )

        acm.recordCountEvent(
            CountEvent(AcmCount.LoadAdSuccess.eventName)
                .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
        )

        originListener?.onAdLoadSuccess(molocoAd)
    }

    override fun onAdLoadFailed(internalError: MolocoInternalAdError) {
        MolocoLogger.info(TAG, "onAdLoadFailed: $internalError")
        provideSdkEvents()?.onAdLoadFailed?.let { url ->
            sdkEventUrlTracker(url, System.currentTimeMillis(), error = internalError)
        }

        acm.recordTimerEvent(
            acmLoadTimerEvent
                .withTag(AcmTag.Result.tagName, "failure")
                .withTag(AcmTag.Reason.tagName, internalError.subErrorType.metricsRepresentation)
                .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
        )

        acm.recordCountEvent(
            CountEvent(AcmCount.LoadAdFailed.eventName)
                .withTag("network", internalError.molocoAdError.networkName)
                .withTag(AcmTag.Reason.tagName, internalError.subErrorType.metricsRepresentation)
                .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
        )

        originListener?.onAdLoadFailed(internalError.molocoAdError)
    }
}
