package com.moloco.sdk.internal.publisher

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.acm.TimerEvent
import com.moloco.sdk.internal.InternalSDKErrorSubType
import com.moloco.sdk.internal.MolocoInternalAdError
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.createInternalAdErrorInfo
import com.moloco.sdk.internal.ortb.model.AutoStore
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.ortb.model.Player
import com.moloco.sdk.internal.ortb.model.SdkEvents
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.AnalyticsApplicationLifecycleTracker
import com.moloco.sdk.publisher.AdFormatType
import com.moloco.sdk.publisher.AdLoad
import com.moloco.sdk.publisher.AdShowListener
import com.moloco.sdk.publisher.MolocoAdError.ErrorType
import com.moloco.sdk.publisher.createAdInfo
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdDisplayState
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AggregatedAdShowListener
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AggregatedFullscreenAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AggregatedOptions
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.FullscreenAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import com.moloco.sdk.acm.AndroidClientMetrics as acm

// TODO. Refactor Duplicate code (BannerImpl)
/**
 * Represents a fullscreen advertisement with various configuration options.
 *
 * @param T The type of listener for ad show events.
 * @param context The application context used for various operations related to the ad.
 * @param appLifecycleTrackerService Service used to track application lifecycle events for analytics.
 * @param customUserEventBuilderService Service used to build custom user events for the ad.
 * @param adUnitId The unique identifier for the ad unit.
 * @param persistentHttpRequest A persistent HTTP request used for loading ad data.
 * @param externalLinkHandler Handler for processing external links within the ad.
 * @param generateAggregatedOptions A lambda function to generate aggregated options based on the player extension.
 * @param adDataHolder Holder for storing ad data, initialized with a default value.
 * @param adFormatType The format type of the ad.
 * @param watermark The watermark to be applied to the ad.
 */
internal class FullscreenAdImpl<in T : AdShowListener>(
        private val context: Context,
        private val appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
        private val customUserEventBuilderService: CustomUserEventBuilderService,
        private val adUnitId: String,
        private val persistentHttpRequest: PersistentHttpRequest,
        private val externalLinkHandler: ExternalLinkHandler,
        private val generateAggregatedOptions: (playerExt: Player?) -> AggregatedOptions,
        private val adDataHolder: FullscreenAdDataHolder<T> = FullscreenAdDataHolder(),
        private val adFormatType: AdFormatType,
        private val watermark: Watermark,
        private val adCreateLoadTimeoutManager: AdCreateLoadTimeoutManager
) : com.moloco.sdk.publisher.FullscreenAd<T>, CreateAdObjectTime by adCreateLoadTimeoutManager {
    private val scope = CoroutineScope(DispatcherProvider().main)
    private val fullscreenAdCreateToLoadTimerEvent: TimerEvent = acm.startTimerEvent(AcmTimer.CreateToLoad.eventName).withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase())
    private var loadToShowTimerEvent: TimerEvent? = null

    val sdkEvents: SdkEvents?
        get() = adDataHolder.sdkEvents

    val bUrlData: BUrlData?
        get() = adDataHolder.bUrlData

    val creativeType: CreativeType?
        get() = adDataHolder.ad?.creativeType

    /**
     * Tells if the fullscreen ad was forcibly closed by the system or of the user
     * Returns true if ad was closed by system and false otherwise; null - no ad to check the status.
     */
    internal val isAdForciblyClosed: Boolean?
        get() = adDataHolder.ad?.isAdForciblyClosed?.value

    private fun destroyAd(sendErrorEvent: MolocoInternalAdError? = null) {
        with(adDataHolder) {
            adDisplayStateJob?.cancel()
            adDisplayStateJob = null
        }

        val isAdDisplaying = adDataHolder.ad?.isAdDisplaying?.value == true

        with(adDataHolder) {
            ad?.destroy()
            ad = null
        }

        val adShowListener = with(adDataHolder) {
            val listener = internalAdShowListener
            internalAdShowListener = null
            listener
        }

        // TODO. Review. Required?
        sendErrorEvent?.let {
            // TODO: https://mlc.atlassian.net/browse/SDK-1729
            adShowListener?.onAdShowFailed(it)
        }
        // TODO. Review. Required?
        // Make sure ad hidden event is sent event when the ad is destroyed during ad display.
        if (isAdDisplaying) adShowListener?.onAdHidden(createAdInfo(adUnitId))

        // TODO.
        //  Quick workaround for when sdkEvents become null too early in FullscreenAd or Banner due to destroyAd() call,
        //  which leads to onAdHidden, onError not track events.
        adDataHolder.sdkEvents = null

        adDataHolder.bUrlData = null


    }

    override fun destroy() {
        scope.cancel()
        destroyAd()
        vastCompletionStatusListener = null
    }

    private val adLoader = AdLoad(
        scope,
        adCreateLoadTimeoutManager::calculateTimeout,
        adUnitId,
        ::recreateXenossAd,
        adFormatType)

    private fun recreateXenossAd(
        bid: Bid
    ): com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad {
        destroyAd()

        options = generateAggregatedOptions(bid.ext?.player)
        autoStore = bid.ext?.player?.autoStore

        val fullscreenAd = AggregatedFullscreenAd(
            context,
            customUserEventBuilderService,
            bid = bid,
            externalLinkHandler = externalLinkHandler,
            watermark = watermark
        )

        with(adDataHolder) {
            ad = fullscreenAd
            sdkEvents = bid.ext?.sdkEvents
            bUrlData = if (bid.burl != null) BUrlData(bid.burl, bid.price) else null
        }

        return fullscreenAd
    }

    override val isLoaded: Boolean
        get() = adLoader.isLoaded

    override fun load(bidResponseJson: String, listener: AdLoad.Listener?) {
        // We do this here instead of AdLoadImpl to be as close to the public API as possible
        acm.recordTimerEvent(fullscreenAdCreateToLoadTimerEvent)
        // Wrapping by launch { ... } block to make sure the fun call is executed on the main thread.
        loadToShowTimerEvent = acm.startTimerEvent(AcmTimer.LoadToShow.eventName)
        scope.launch {
            adLoader.load(bidResponseJson, listener)
        }
    }

    private var options: AggregatedOptions = generateAggregatedOptions(null)
    private var autoStore: AutoStore? = null


    override fun show(listener: T?) {
        loadToShowTimerEvent?.let {
            acm.recordTimerEvent(it
                .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase()))
        }
        acm.recordCountEvent(CountEvent(AcmCount.ShowAdAttempt.eventName)
            .withTag(AcmTag.AdType.tagName, adFormatType.name.lowercase()))
        // Wrapping by launch { ... } block to make sure the fun call is executed on the main thread.
        scope.launch {
            if (listener != null) {
                adDataHolder.internalAdShowListener = InternalInterstitialAdShowListenerTracker(
                    listener,
                    appLifecycleTrackerService,
                    customUserEventBuilderService,
                    { sdkEvents },
                    { bUrlData },
                    adFormatType)
            } else {
                adDataHolder.internalAdShowListener = null
            }
            val internalListener = adDataHolder.internalAdShowListener

            val ad = adDataHolder.ad
            if (ad == null || !isLoaded) {
                internalListener?.onAdShowFailed(
                    createInternalAdErrorInfo(adUnitId, ErrorType.AD_SHOW_ERROR_NOT_LOADED, InternalSDKErrorSubType.AD_SHOW_ERROR_NOT_LOADED))
                return@launch
            }

            if (ad.isAdDisplaying.value) {
                internalListener?.onAdShowFailed(
                    createInternalAdErrorInfo(adUnitId, ErrorType.AD_SHOW_ERROR_ALREADY_DISPLAYING, InternalSDKErrorSubType.AD_SHOW_ERROR_ALREADY_DISPLAYING))
                return@launch
            }

            ad.listenToAdDisplayState(internalListener)

            ad.show(options, provideAdShowListener(internalListener))
        }
    }

    private fun AdDisplayState.listenToAdDisplayState(adShowListener: InternalAdShowListener?) {
        with(adDataHolder) {
            adDisplayStateJob?.cancel()
            adDisplayStateJob = scope.launch {
                isAdDisplaying.first { it }
                adShowListener?.onAdShowSuccess(createAdInfo(adUnitId))
                isAdDisplaying.first { !it }
                adShowListener?.onAdHidden(createAdInfo(adUnitId))
            }
        }
    }

    var vastCompletionStatusListener: ((skipped: Boolean) -> Unit)? = null

    private fun provideAdShowListener(adShowListener: InternalAdShowListener?) = object : AggregatedAdShowListener {

        override fun onVastCompletionStatus(skipped: Boolean) {
            autoStore?.let {
                if (it.enabled && (!skipped || it.onSkip)) {
                    it.eventLink?.let { eventLink ->
                        persistentHttpRequest.send(eventLink)
                    }
                }
            }

            vastCompletionStatusListener?.invoke(skipped)
        }

        override fun onShowError(internalShowError: MolocoAdSubErrorType) {
            destroyAd(sendErrorEvent = createInternalAdErrorInfo(adUnitId, ErrorType.AD_SHOW_ERROR, internalShowError))
        }

        override fun onClick() {
            adShowListener?.onAdClicked(createAdInfo(adUnitId))
        }
    }
}

@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal class FullscreenAdDataHolder<T : AdShowListener>(
    var ad: FullscreenAd<AggregatedAdShowListener, AggregatedOptions>? = null,
    var sdkEvents: SdkEvents? = null,
    var bUrlData: BUrlData? = null,
    var adDisplayStateJob: Job? = null,
    var internalAdShowListener: InternalAdShowListener? = null
)
