package com.moloco.sdk.internal.publisher.nativead

import android.annotation.SuppressLint
import android.content.Context
import android.view.View
import android.view.ViewGroup
import android.view.ViewGroup.LayoutParams.MATCH_PARENT
import android.widget.FrameLayout
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.ComposeView
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.TimerEvent
import com.moloco.sdk.internal.ViewLifecycleOwner
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.ortb.model.Bid
import com.moloco.sdk.internal.ortb.model.SdkEvents
import com.moloco.sdk.internal.publisher.AdCreateLoadTimeoutManager
import com.moloco.sdk.internal.publisher.AdLoad
import com.moloco.sdk.internal.publisher.BUrlData
import com.moloco.sdk.internal.publisher.CreateAdObjectTime
import com.moloco.sdk.internal.publisher.InternalAdShowListener
import com.moloco.sdk.internal.publisher.InternalBannerAdShowListenerTracker
import com.moloco.sdk.internal.publisher.nativead.nativeadviewprovider.NativeAdComposeViewWrapper
import com.moloco.sdk.internal.publisher.nativead.nativeadviewprovider.provideCTAText
import com.moloco.sdk.internal.publisher.nativead.nativeadviewprovider.provideDescription
import com.moloco.sdk.internal.publisher.nativead.nativeadviewprovider.provideIcon
import com.moloco.sdk.internal.publisher.nativead.nativeadviewprovider.provideMainImage
import com.moloco.sdk.internal.publisher.nativead.nativeadviewprovider.provideRating
import com.moloco.sdk.internal.publisher.nativead.nativeadviewprovider.provideSponsored
import com.moloco.sdk.internal.publisher.nativead.nativeadviewprovider.provideTitle
import com.moloco.sdk.internal.publisher.nativead.ui.NativeVideoPlaybackControlUI
import com.moloco.sdk.internal.publisher.nativead.ui.nativeVideoMuteButton
import com.moloco.sdk.internal.services.AnalyticsApplicationLifecycleTracker
import com.moloco.sdk.internal.services.AudioService
import com.moloco.sdk.publisher.AdLoad
import com.moloco.sdk.publisher.NativeAdForMediation
import com.moloco.sdk.publisher.NativeAdOrtbRequestRequirements
import com.moloco.sdk.publisher.createAdInfo
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.NativeAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ViewVisibilityTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.vastAdPlaylistController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.OverrideVastContainerOnClick
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.VastRenderer
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.defaultPlaybackControl
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.cancel
import kotlin.time.DurationUnit
import kotlin.time.toDuration
import com.moloco.sdk.acm.AndroidClientMetrics as acm

internal class NativeAdForMediationImpl(
    private val context: Context,
    private val appLifecycleTrackerService: AnalyticsApplicationLifecycleTracker,
    private val customUserEventBuilderService: CustomUserEventBuilderService,
    private val audioService: AudioService,
    private val adUnitId: String,
    private val viewVisibilityTracker: ViewVisibilityTracker,
    private val externalLinkHandler: ExternalLinkHandler,
    private val persistentHttpRequest: PersistentHttpRequest,
    override val nativeAdOrtbRequestRequirements: NativeAdOrtbRequestRequirements.Requirements,
    // TODO: Once Native ads is re-done, this needs to be injected
    private val createLoadTimeoutManager: AdCreateLoadTimeoutManager,
    private val viewLifecycleOwnerSingleton: ViewLifecycleOwner,
) : NativeAdForMediation, CreateAdObjectTime by createLoadTimeoutManager {

    private val nativeAdForMediationCreateToLoadTimerEvent: TimerEvent = AndroidClientMetrics.startTimerEvent(
        AcmTimer.CreateToLoad.eventName).withTag(AcmTag.AdType.tagName, getAdFormatType(nativeAdOrtbRequestRequirements).name.lowercase())
    private val adLoadScope = MainScope()

    private val adLoader = AdLoad(
        adLoadScope,
        createLoadTimeoutManager::calculateTimeout,
        adUnitId,
        ::recreateXenossNativeAd,
        getAdFormatType(nativeAdOrtbRequestRequirements))

    override val isLoaded: Boolean by adLoader::isLoaded

    override fun load(bidResponseJson: String, listener: AdLoad.Listener?) {
        // Recording of this timer is as close to the public API as we can get.
        acm.recordTimerEvent(nativeAdForMediationCreateToLoadTimerEvent)
        adLoader.load(bidResponseJson, listener)
    }

    // TODO. Mostly Duplicate of NativeBannerImpl, BannerImpl.
    private fun recreateXenossNativeAd(
        bid: Bid
    ): com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.AdLoad {
        destroyAd()

        val nativeAdScope = MainScope()
        this.nativeAdScope = nativeAdScope

        val nativeAd = NativeAd(
            context,
            bid.adm,
            nativeAdScope,
            externalLinkHandler,
            persistentHttpRequest = persistentHttpRequest,
        )
        this.nativeAd = nativeAd

        sdkEvents = bid.ext?.sdkEvents
        bUrlData = if (bid.burl != null) BUrlData(bid.burl, bid.price) else null

        adShowListenerTracker = InternalBannerAdShowListenerTracker(
            originListener = null,
            appLifecycleTrackerService,
            customUserEventBuilderService,
            provideSdkEvents = { sdkEvents },
            provideBUrlData = { bUrlData },
            adType = getAdFormatType(nativeAdOrtbRequestRequirements)
        )

        return nativeAd
    }

    override var interactionListener: NativeAdForMediation.InteractionListener? = null

    override fun handleImpression() {
        nativeAd?.trackImpression()
        interactionListener?.onImpressionHandled()
        adShowListenerTracker?.onAdShowSuccess(createAdInfo(adUnitId))
    }

    override fun handleGeneralAdClick() {
        handleAssetClick(null)
    }

    override fun handleIconClick() {
        handleAssetClick(NativeAdAssetIds.ICON)
    }

    override fun handleMainImageClick() {
        handleAssetClick(NativeAdAssetIds.MAIN_IMAGE)
    }

    private fun handleVideoViewClick() {
        handleAssetClick(NativeAdAssetIds.VIDEO)
    }

    private fun handleAssetClick(assetId: Int?) {
        if (nativeAd?.handleAssetClick(assetId) == true) {
            interactionListener?.onGeneralClickHandled()
            adShowListenerTracker?.onAdClicked(createAdInfo(adUnitId))
        }
    }

    override val title: String?
        get() = nativeAd?.preparedAssets?.provideTitle({})?.text
    override val description: String?
        get() = nativeAd?.preparedAssets?.provideDescription({})?.text
    override val sponsorText: String?
        get() = nativeAd?.preparedAssets?.provideSponsored({})?.text
    override val callToActionText: String?
        get() = nativeAd?.preparedAssets?.provideCTAText({})?.text
    override val rating: Float?
        get() = nativeAd?.preparedAssets?.provideRating({})?.stars
    override val iconUri: String?
        get() = nativeAd?.preparedAssets?.provideIcon({})?.uri
    override val mainImageUri: String?
        get() = nativeAd?.preparedAssets?.provideMainImage({})?.uri

    private var vastAdController: AdController? = null
    private var _video: VideoContainer? = null
    override val video: View?
        get() {
            if (_video != null) return _video

            val videoAsset =
                nativeAd?.preparedAssets?.videos?.get(NativeAdAssetIds.VIDEO)
                    ?: return null

            val vastAdController = vastAdPlaylistController(
                videoAsset.vastAd,
                externalLinkHandler,
                context,
                mute = audioService.isMediaVolumeOnMute,
                overrideLinearGoNextActionEnabled = false,
                overrideLinearGoNextActionEnabledDelaySeconds = 0,
                companionGoNextActionDelaySeconds = 0,
                decGoNextActionDelaySeconds = 0,
                autoStoreOnComplete = false,
                autoStoreOnSkip = false,
                customUserEventBuilderService = customUserEventBuilderService
            )
            this.vastAdController = vastAdController

            vastAdController.show()

            _video = VideoContainer(context, vastAdController, viewVisibilityTracker, viewLifecycleOwnerSingleton, ::handleVideoViewClick)
            return _video
        }

    private fun destroyVideo() {
        _video?.destroyUnderlyingVideoComposeView()
        _video = null

        vastAdController?.destroy()
        vastAdController = null
    }

    private var nativeAdScope: CoroutineScope? = null
    private var nativeAd: NativeAd? = null
    private var adShowListenerTracker: InternalAdShowListener? = null
    private var sdkEvents: SdkEvents? = null
    private var bUrlData: BUrlData? = null

    private fun destroyAd() {
        destroyVideo()

        nativeAdScope?.cancel()
        nativeAdScope = null

        nativeAd = null

        adShowListenerTracker = null

        // TODO. Also duplicate of BannerImpl, FullscreenAd.
        //  Quick workaround for when sdkEvents become null too early in FullscreenAd or Banner due to destroyAd() call,
        //  which leads to onAdHidden, onError not track events.
        sdkEvents = null
        bUrlData = null
    }

    override fun destroy() {
        adLoadScope.cancel()
        destroyAd()
        interactionListener = null
    }

    companion object {
        /**
         * For those wondering how we came to this number...
         * It took planning and guile. Some sleepless nights and the moto, try hard or die trying.
         * Finally we arrived at the conclusion:
         * Applovin MAX has a 10 seconds timeout. We subtract 1 second to avoid timeout race condition.
         */
        internal val LOAD_TIMEOUT = 9.toDuration(DurationUnit.SECONDS)
    }
}

// https://mlc.atlassian.net/browse/SDK-924
// View.applyVanillaActivityAndJetpackComposeCompatibility() HAS TO be called on the WRAPPING CONTAINER view,
// otherwise it'll be too late and we get "ViewTreelifecycleOwner not found" exception;
// ComposeView immediately checks for missing essential components in WindowAttached event handler.
// When applyVanillaActivityAndJetpackComposeCompatibility() called on the wrapping container,
// the implementation of the function injects those essential components (if missing) during Wrapper container's window attach event,
// since it's the event where it's possible to get rootView and apply the fix properly.
// Considering wrapper container's window attach event happens before inner ComposeView one, we're right on time for applying the patch,
// so that in the following ComposeView's window attach event, the check for missing components passes.
@SuppressLint("ViewConstructor")
private class VideoContainer(
        context: Context,
        vastAdController: AdController,
        viewVisibilityTracker: ViewVisibilityTracker,
        viewLifecycleOwnerSingleton: ViewLifecycleOwner,
        onClick: () -> Unit
) : FrameLayout(context) {

    init {
        viewLifecycleOwnerSingleton.addLifecycleOwnerSupportTo(this)
    }

    private var videoView: ComposeView? = NativeAdComposeViewWrapper(context) { modifier ->
        VastRenderer(
            modifier = modifier,
            adViewModel = vastAdController,
            backgroundColor = Color.Black,
            replayButton = null,
            adCloseCountdownButton = null,
            adSkipCountdownButton = null,
            ctaButton = null,
            overrideVastContainerOnClick = OverrideVastContainerOnClick(
                overrideLinearOnClick = onClick,
                overrideCompanionOnClick = onClick,
                overrideDECOnClick = onClick
            ),
            progressBar = null,
            muteButton = nativeVideoMuteButton(),
            playbackControl = defaultPlaybackControl(
                NativeVideoPlaybackControlUI
            ),
            viewVisibilityTracker =  viewVisibilityTracker
        )
    }.also {
        addView(it, ViewGroup.LayoutParams(MATCH_PARENT, MATCH_PARENT))
    }

    fun destroyUnderlyingVideoComposeView() {
        removeAllViews()
        videoView?.disposeComposition()
        videoView = null
    }
}
