package com.moloco.sdk.internal.services

import android.content.ContentValues.TAG
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.booleanPreferencesKey
import androidx.datastore.preferences.core.doublePreferencesKey
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.floatPreferencesKey
import androidx.datastore.preferences.core.intPreferencesKey
import androidx.datastore.preferences.core.longPreferencesKey
import androidx.datastore.preferences.core.stringPreferencesKey
import com.moloco.sdk.internal.MolocoLogger
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.map

/**
 * SDK wide key/value data store that provides type safe reads/writes
 */
internal interface DataStoreService {
    suspend fun getInt(key: String): Int?
    suspend fun getString(key: String): String?
    suspend fun getBoolean(key: String): Boolean?
    suspend fun getFloat(key: String): Float?
    suspend fun getDouble(key: String): Double?
    suspend fun getLong(key: String): Long?

    suspend fun removeInt(key: String)
    suspend fun removeString(key: String)
    suspend fun removeFloat(key: String)
    suspend fun removeDouble(key: String)
    suspend fun removeLong(key: String)
    suspend fun removeBoolean(key: String)

    suspend fun <T> set(key: String, newValue: T)
}

internal class PreferencesDataStoreServiceImpl(private val dataStore: DataStore<Preferences>) : DataStoreService {
    override suspend fun getInt(key: String): Int? {
        return dataStore.data.map { preferences ->
            preferences[intPreferencesKey(key)]
        }.firstOrNull()
    }

    override suspend fun getString(key: String): String? {
        return dataStore.data.map { preferences: Preferences ->
            preferences[stringPreferencesKey(key)]
        }.firstOrNull()
    }

    override suspend fun getBoolean(key: String): Boolean? {
        return dataStore.data.map { preferences ->
            preferences[booleanPreferencesKey(key)]
        }.firstOrNull()
    }

    override suspend fun getFloat(key: String): Float? {
        return dataStore.data.map { preferences ->
            preferences[floatPreferencesKey(key)]
        }.firstOrNull()
    }

    override suspend fun getDouble(key: String): Double? {
        return dataStore.data.map { preferences ->
            preferences[doublePreferencesKey(key)]
        }.firstOrNull()
    }

    override suspend fun getLong(key: String): Long? {
        return dataStore.data.map { preferences ->
            preferences[longPreferencesKey(key)]
        }.firstOrNull()
    }

    override suspend fun removeInt(key: String) {
        dataStore.edit { settings ->
            settings.remove(intPreferencesKey(key))
        }
    }

    override suspend fun removeString(key: String) {
        dataStore.edit { settings ->
            settings.remove(stringPreferencesKey(key))
        }
    }

    override suspend fun removeFloat(key: String) {
        dataStore.edit { settings ->
            settings.remove(floatPreferencesKey(key))
        }
    }

    override suspend fun removeDouble(key: String) {
        dataStore.edit { settings ->
            settings.remove(doublePreferencesKey(key))
        }
    }

    override suspend fun removeLong(key: String) {
        dataStore.edit { settings ->
            settings.remove(longPreferencesKey(key))
        }
    }

    override suspend fun removeBoolean(key: String) {
        dataStore.edit { settings ->
            settings.remove(booleanPreferencesKey(key))
        }
    }

    override suspend fun <T> set(key: String, newValue: T) {
        when (newValue) {
            is Int -> typeSet(intPreferencesKey(key), newValue)
            is String -> typeSet(stringPreferencesKey(key), newValue)
            is Float -> typeSet(floatPreferencesKey(key), newValue)
            is Double -> typeSet(doublePreferencesKey(key), newValue)
            is Long -> typeSet(longPreferencesKey(key), newValue)
            is Boolean -> typeSet(booleanPreferencesKey(key), newValue)
            else -> {
                MolocoLogger.warn(TAG, "Unexpected value type: $newValue for key: $key")
            }
        }
    }

    private suspend inline fun <reified T : Any> typeSet(key: Preferences.Key<T>, value: T) {
        dataStore.edit { settings ->
            settings[key] = value
        }
    }
}
