package com.moloco.sdk.internal.services

import android.content.Context
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.Build
import androidx.annotation.RequiresApi
import com.moloco.sdk.internal.services.NetworkInfo.CellularNetwork
import com.moloco.sdk.internal.services.NetworkInfo.NoNetwork
import com.moloco.sdk.internal.services.NetworkInfo.WifiNetwork

internal interface NetworkInfoService {
    suspend operator fun invoke(): NetworkInfo
}

internal sealed class NetworkInfo {
    object NoNetwork : NetworkInfo()
    object WifiNetwork : NetworkInfo()
    class CellularNetwork(val carrier: String) : NetworkInfo()
}

/**
 * Implementation of [NetworkInfoService] responsible for providing information about the network connectivity.
 *
 * @property context The application context.
 * @property deviceInfoService The service providing device information.
 */
internal class NetworkInfoServiceImpl(
    private val context: Context,
    private val deviceInfoService: DeviceInfoService,
) : NetworkInfoService {

    /**
     * Retrieves network information asynchronously.
     *
     * @return [NetworkInfo] representing the current network state.
     */
    override suspend fun invoke(): NetworkInfo {
        val manager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            manager.networkInfoApi23Plus()
        } else {
            manager.networkInfo()
        }
    }

    /**
     * Retrieves network information for devices running API level 23 (Android M) or higher.
     *
     * @return [NetworkInfo] representing the current network state.
     */
    @RequiresApi(Build.VERSION_CODES.M)
    private suspend fun ConnectivityManager.networkInfoApi23Plus(): NetworkInfo {
        if (activeNetwork == null) return NoNetwork
        val networkCapabilities = getNetworkCapabilities(activeNetwork) ?: return NoNetwork

        return when {
            networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) -> WifiNetwork
            networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) -> CellularNetwork(deviceInfoService().mobileCarrier)
            else -> NoNetwork
        }
    }

    /**
     * Retrieves network information for devices running below API level 23 (Android M).
     *
     * @return [NetworkInfo] representing the current network state.
     */
    @Suppress("DEPRECATION")
    private suspend fun ConnectivityManager.networkInfo(): NetworkInfo = when (activeNetworkInfo?.type) {
        ConnectivityManager.TYPE_WIFI -> WifiNetwork
        ConnectivityManager.TYPE_MOBILE -> CellularNetwork(deviceInfoService().mobileCarrier)
        else -> NoNetwork
    }
}

