package com.moloco.sdk.internal.services.bidtoken

import com.google.protobuf.ByteString
import com.moloco.sdk.BidToken
import com.moloco.sdk.internal.services.DeviceInfo
import com.moloco.sdk.internal.services.ScreenInfo
import com.moloco.sdk.publisher.privacy.MolocoPrivacy
import com.moloco.sdk.service_locator.SdkObjectFactory
import java.util.Date
import java.util.TimeZone

/**
 * Client side token builder as per the proposal here - https://docs.google.com/document/d/1elJPg3JNxJ4s6qo1OdHDlo6mOgTNtBOHwctkJDl5r4o/edit#heading=h.rvg2y6iz8zo9
 */
internal interface ClientBidTokenBuilder {
    companion object {
        fun create(): ClientBidTokenBuilder {
            return ClientBidTokenBuilderImpl(
                SdkObjectFactory.DeviceAndApplicationInfo.deviceInfoSingleton(),
                SdkObjectFactory.DeviceAndApplicationInfo.screenInfoSingleton())
        }
    }
    fun buildBidToken(bidTokenComponents: ByteArray, secret: ByteArray): ByteArray
    fun buildClientBidTokenComponent(privacySettings: MolocoPrivacy.PrivacySettings, bidTokenConfig: BidTokenConfig): BidToken.ClientBidTokenComponents

    fun parseClientBidTokenComponent(payload: ByteArray): BidToken.ClientBidTokenComponents
    fun parse(payload: ByteArray): BidToken.ClientBidToken
}

internal class ClientBidTokenBuilderImpl(
    private val deviceInfo: DeviceInfo,
    private val screenInfo: ScreenInfo,
): ClientBidTokenBuilder {
    override fun buildBidToken(
        bidTokenComponents: ByteArray,
        secret: ByteArray,
    ): ByteArray {
        return BidToken.ClientBidToken.newBuilder().apply {
            es = ByteString.copyFrom(secret)
            payload = ByteString.copyFrom(bidTokenComponents)
        }.build().toByteArray()
    }

    override fun buildClientBidTokenComponent(
        privacySettings: MolocoPrivacy.PrivacySettings,
        bidTokenConfig: BidTokenConfig
    ): BidToken.ClientBidTokenComponents {
        return BidToken.ClientBidTokenComponents.newBuilder().apply {
            privacy = BidToken.ClientBidTokenComponents.Privacy.newBuilder().apply {
                privacySettings.isAgeRestrictedUser?.let {
                    coppa = it
                }
                privacySettings.isUserConsent?.let {
                    gdpr = it
                }
                privacySettings.isDoNotSell?.let {
                    ccpa = it
                }
                privacySettings.TCFConsent?.let {
                    tcfConsentString = it
                }
                usPrivacy = privacySettings.usPrivacy
            }.build()

            device = BidToken.ClientBidTokenComponents.Device.newBuilder().apply {
                // 1
                language = deviceInfo.language
                // 2
                osv = deviceInfo.osVersion
                // 3
                make = deviceInfo.manufacturer
                // 4
                model = deviceInfo.model
                // 5
                hwv = deviceInfo.hwVersion
                // 6
                carrier = deviceInfo.mobileCarrier
                // 7
                devicetype = if (deviceInfo.isTablet) 5 else 1
                // 8
                js = 1
                // 9
                geo = BidToken.ClientBidTokenComponents.Geo.newBuilder().apply {
                    utcoffset = TimeZone.getDefault().getOffset(Date().time) / 60000
                }.build()
                // 10
                w = screenInfo.screenWidthPx
                // 11
                h = screenInfo.screenHeightPx
                // 12
                pxratio = screenInfo.density.toDouble()
                // 13
                ppi = screenInfo.dpi
                // 14
                os = deviceInfo.os
                // 15
                if (bidTokenConfig.dbtEnabled) {
                    dbt = deviceInfo.dbtMs.millisToNanos()
                }
            }.build()
        }.build()

    }

    override fun parse(payload: ByteArray): BidToken.ClientBidToken {
        return BidToken.ClientBidToken.parseFrom(payload)
    }

    override fun parseClientBidTokenComponent(payload: ByteArray): BidToken.ClientBidTokenComponents {
        return BidToken.ClientBidTokenComponents.parseFrom(payload)
    }

    private fun Long.millisToNanos(): Long = this * 1_000_000
}