package com.moloco.sdk.internal.services.bidtoken

import android.util.Base64
import com.moloco.sdk.BuildConfig
import com.moloco.sdk.acm.AndroidClientMetrics
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.client_metrics_data.AcmCount
import com.moloco.sdk.internal.client_metrics_data.AcmResultTag
import com.moloco.sdk.internal.client_metrics_data.AcmTag
import com.moloco.sdk.internal.client_metrics_data.AcmTimer
import com.moloco.sdk.internal.services.encryption.EncryptionService
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import java.security.InvalidAlgorithmParameterException
import java.security.InvalidKeyException
import java.security.NoSuchAlgorithmException
import java.security.spec.InvalidKeySpecException
import javax.crypto.BadPaddingException
import javax.crypto.IllegalBlockSizeException
import javax.crypto.NoSuchPaddingException

/**
 * Client side bid token provider as per the proposal here:
 * https://docs.google.com/document/d/1elJPg3JNxJ4s6qo1OdHDlo6mOgTNtBOHwctkJDl5r4o/edit#heading=h.rvg2y6iz8zo9
 */
internal interface ClientBidTokenService {
    companion object {
        fun create(): ClientBidTokenService =
            ClientBidTokenServiceImpl(
                clientBidTokenBuilder = ClientBidTokenBuilder.create(),
                encryptionService = EncryptionService.create(),
                privacyProvider = MolocoPrivacyBasedPrivacyProvider(),
            )
    }

    suspend fun bidToken(publicKey: String, bidTokenConfig: BidTokenConfig): Result<String>
}

internal class ClientBidTokenServiceImpl(
    private val clientBidTokenBuilder: ClientBidTokenBuilder,
    private val encryptionService: EncryptionService,
    private val privacyProvider: PrivacyProvider,
) : ClientBidTokenService {
    private val TAG = "ClientBidTokenServiceImpl"
    private var rsaPublicKey: String = ""
    private var clientBidToken: String = ""
    private var config: BidTokenConfig = DefaultBidTokenConfig

    private val mutex = Mutex()
    // Initiating to have privacy to compare to upon bidToken() invocations.
    private var privacy = privacyProvider.privacy

    override suspend fun bidToken(publicKey: String, bidTokenConfig: BidTokenConfig): Result<String> {
        mutex.withLock {
            if (shouldRefreshClientBidToken(publicKey, bidTokenConfig)) {
                logD("Bid token needs refresh")
                rsaPublicKey = publicKey
                config = bidTokenConfig
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ClientBidTokenCached.eventName).withTag(AcmTag.Result.tagName, "false"))
                clientBidToken = clientBidToken(publicKey)
            } else {
                logD("Bid token doesn't need refresh")
                AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ClientBidTokenCached.eventName).withTag(AcmTag.Result.tagName, "true"))
            }

            if (clientBidToken.isEmpty()) {
                return Result.failure(Exception("Client bid token is empty"))
            }

            return Result.success(clientBidToken)
        }
    }

    /**
     * @return true - privacy updated; false - privacy didn't change.
     */
    private fun tryUpdatePrivacy(): Boolean {
        val oldPrivacy = privacy
        val newPrivacy = privacyProvider.privacy
        privacy = newPrivacy

        return (oldPrivacy != newPrivacy).also {
            logD(if (it) "privacy updated" else "privacy didn't change")
        }
    }

    private fun tryUpdateConfig(newConfig: BidTokenConfig): Boolean {
        val oldConfig = config
        config = newConfig

        return (oldConfig != newConfig).also {
            logD(if (it) "config updated" else "config didn't change")
        }
    }

    private fun shouldRefreshClientBidToken(publicKey: String, config: BidTokenConfig): Boolean {
        val wasPrivacyUpdated = tryUpdatePrivacy()
        if (wasPrivacyUpdated) {
            logD( "privacy was updated, needs refresh")
            return true
        }

        val rsaPublicKeyUpdated = (rsaPublicKey != publicKey)
        if (rsaPublicKeyUpdated) {
            return true
        }

        val configUpdated = tryUpdateConfig(config)
        if (configUpdated) {
            return true
        }

        if (clientBidToken.isEmpty()) {
            logD( "cached bidToken is empty, needs refresh")
            return true
        }

        logD("Bid token doesn't need refresh")
        return false
    }

    private fun clientBidToken(publicKey: String): String {
        if (publicKey.isEmpty()) {
            AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ClientBidTokenBuild.eventName).withTag(AcmTag.Result.tagName, AcmResultTag.failure.name).withTag(AcmTag.Reason.tagName, "empty_public_key"))
            return ""
        }

        val buildTimerEvent = AndroidClientMetrics.startTimerEvent(AcmTimer.ClientBidTokenBuild.eventName)
        val failureReason: String
        try {
            val encryptedAesKey = encryptionService.rsaEncryptedAesKey(publicKey)
            val clientBidTokenComponent = clientBidTokenBuilder.buildClientBidTokenComponent(privacy, config)
            val base64clientBidTokenComponent = Base64.encode(clientBidTokenComponent.toByteArray(), Base64.DEFAULT)
            val encryptedClientBidToken = encryptionService.aesEncrypt(base64clientBidTokenComponent)
            val base64BidToken = Base64.encode(encryptedClientBidToken, Base64.DEFAULT)
            val clientBidToken = clientBidTokenBuilder.buildBidToken(base64BidToken, encryptedAesKey)
            val base64EncodedClientBidTokenPayload = Base64.encodeToString(clientBidToken, Base64.DEFAULT)

            AndroidClientMetrics.recordTimerEvent(buildTimerEvent.withTag(AcmTag.Result.tagName, AcmResultTag.success.name))
            AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ClientBidTokenBuild.eventName).withTag(AcmTag.Result.tagName, AcmResultTag.success.name))
            return "v1:$base64EncodedClientBidTokenPayload"
        } catch (nsae: NoSuchAlgorithmException) {
            failureReason = "no_such_algorithm_exception"
        } catch (nspe: NoSuchPaddingException) {
            failureReason = "no_such_padding_exception"
        } catch (ike: InvalidKeyException) {
            failureReason = "invalid_key_exception"
        } catch (ibse: IllegalBlockSizeException) {
            failureReason = "illegal_block_size_exception"
        } catch(bpe: BadPaddingException) {
            failureReason = "bad_padding_exception"
        } catch(iape: InvalidAlgorithmParameterException) {
            failureReason = "invalid_algorithm_parameter_exception"
        } catch(ikse: InvalidKeySpecException) {
            failureReason = "invalid_key_spec_exception"
        } catch (iae: IllegalArgumentException) {
            failureReason = "illegal_argument_exception"
        } catch (e: Exception) {
            failureReason = "unknown_exception"
        }

        AndroidClientMetrics.recordCountEvent(CountEvent(AcmCount.ClientBidTokenBuild.eventName).withTag(AcmTag.Result.tagName, AcmResultTag.failure.name).withTag(AcmTag.Reason.tagName, failureReason))
        AndroidClientMetrics.recordTimerEvent(buildTimerEvent.withTag(AcmTag.Result.tagName, AcmResultTag.failure.name).withTag(AcmTag.Reason.tagName, failureReason))
        return ""
    }

    // We don't want to log our bid token in release variant. Perhaps this can be supported at MolocoLogger level
    private fun logD(message: String) {
        if (BuildConfig.DEBUG) {
            MolocoLogger.debug(TAG, message)
        }
    }
}
